<?php

namespace Drupal\beehotel_vertical\Controller;

use Drupal\bee_hotel\Util\Event;
use Drupal\bee_hotel\Util\BeeHotelUnit;
use Drupal\beehotel_vertical\BeehotelVertical;
use CommerceGuys\Intl\Formatter\CurrencyFormatterInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBuilder;
use Drupal\Core\Link;
use Drupal\Core\Render\Markup;
use Drupal\Core\Render\RendererInterface;
use Drupal\Component\Utility\Unicode;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides route responses for BeeHotel module.
 */
class Vertical extends ControllerBase {

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilder
   */
  protected $formBuilder;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The currency formatter.
   *
   * @var \CommerceGuys\Intl\Formatter\CurrencyFormatterInterface
   */
  protected $currencyFormatter;

  /**
   * Representation of the current HTTP request.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The bee hotel event.
   *
   * @var \Drupal\bee_hotel\Util\Event
   */
  private $beehotelEvent;

  /**
   * The bee hotel unit.
   *
   * @var \Drupal\bee_hotel\Util\BeeHotelUnit
   */
  private $beehotelUnit;

  /**
   * The BeeHotel vertical time range.
   *
   * @var \Drupal\beehotel_vertical\BeehotelVertical
   */
  private $beehotelVertical;

  /**
   * Constructs a new Vertical object.
   *
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Form\FormBuilder $form_builder
   *   The form builder.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \CommerceGuys\Intl\Formatter\CurrencyFormatterInterface $currency_formatter
   *   The currency formatter.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param Drupal\bee_hotel\Util\Event $beehotel_event
   *   The Bee Hotel Event util.
   * @param \Drupal\bee_hotel\Util\BeeHotelUnit $bee_hotel_unit
   *   The BeeHotel Unit Utility.
   * @param \Drupal\beehotel_vertical\BeehotelVertical $beehotel_vertical
   *   The BeeHotel Vertical Class for features.
   */
  public function __construct(RendererInterface $renderer, FormBuilder $form_builder, DateFormatterInterface $date_formatter, EntityTypeManagerInterface $entity_type_manager, CurrencyFormatterInterface $currency_formatter, RequestStack $request_stack, ConfigFactoryInterface $config_factory, Event $beehotel_event, BeeHotelUnit $bee_hotel_unit, BeehotelVertical $beehotel_vertical) {
    $this->renderer = $renderer;
    $this->formBuilder = $form_builder;
    $this->dateFormatter = $date_formatter;
    $this->entityTypeManager = $entity_type_manager;
    $this->currencyFormatter = $currency_formatter;
    $this->requestStack = $request_stack;
    $this->config = $config_factory;
    $this->eventUtil = $beehotel_event;
    $this->beehotelUnit = $bee_hotel_unit;
    $this->beehotelVertical = $beehotel_vertical;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('renderer'),
      $container->get('form_builder'),
      $container->get('date.formatter'),
      $container->get('entity_type.manager'),
      $container->get('commerce_price.currency_formatter'),
      $container->get('request_stack'),
      $container->get('config.factory'),
      $container->get('bee_hotel.util.event'),
      $container->get('bee_hotel.util.beehotelunit'),
      $container->get('beehotel_vertical.beehotelvertical'),
    );
  }

  /**
   * Produce the table.
   */
  public function result($data) {

    $data['rows'] = $this->rows($data);
    $data['header'] = $this->header($data);

    $output['table'] = [
      '#type' => 'table',
      '#attributes' => [
        'class' => ['vertical-table'],
      ],
      '#prefix' => $this->displayTimeRangeForm(),
      '#header' => $data['header'],
      '#rows' => $data['rows'],
      '#empty' => $this->t('Your table is empty'),
    ];
    return $this->renderer->render($output);
  }

  /**
   * Produce the table header.
   */
  private function header($data) {
    $data['units'] = $this->beehotelUnit->getBeeHotelUnits($options = []);
    $header = [];
    $url = Url::fromRoute('beehotel_vertical.admin_settings', []);
    $link = Link::fromTextAndUrl($this->t('Day'), $url);
    $link = $link->toRenderable();
    $link['#attributes'] = ['class' => ['button', 'gear']];

    $header[] = Markup::create("<h3 class='settings'>" . $this->renderer->render($link) . "</strong>");

    if (!empty($data['units'])) {
      foreach ($data['units'] as $unit) {
        $url = Url::fromRoute('entity.node.edit_form', ['node' => $unit->Id()]);
        $unit_link = Link::fromTextAndUrl($unit->GetTitle(), $url);
        $unit_link = $unit_link->toRenderable();
        $unit_link['#attributes'] = ['class' => ['button', 'link']];
        $header[] = Markup::create("<h3 class='unit'>" . $this->renderer->render($unit_link) . "</h3>");
      }
    }
    $data['header'] = $header;
    return $data['header'];
  }

  /**
   * Add result rows to data.
   *
   * @param array $data
   *   An array with report related data.
   */
  private function rows(array $data) {

    $data['rows'] = [];
    $data['system_email'] = $this->config->get('system.site')->get('mail');

    $config = $this->config->get('beehotel_vertical.settings');
    $timejump = $config->get('vertical.timejump');
    $data['rowsnumber'] = $this->rowsNumber();

    $data['units'] = $this->beehotelUnit->getBeeHotelUnits($options = []);

    // Produce a row for every requested day.
    for ($i = 0; $i <= $data['rowsnumber']; $i++) {
      $data['day']['timestamp'] = (time() - $timejump) + ($i * (60 * 60 * 24));

      // @todo move this into a twig template
      $data['day']['formatted'] =
        "<span class='day-of-the-week'>" .

          $this->dateFormatter->format($data['day']['timestamp'], 'custom', 'D') . ",
        </span>" .

        "<span class='day-month'>" .
         $this->dateFormatter->format($data['day']['timestamp'], 'custom', 'd M') . ",
        </span>" .

        "<span class='year'>" .
         $this->dateFormatter->format($data['day']['timestamp'], 'custom', 'Y') . "
        </span>";

      $data['day']['d'] = $this->dateFormatter->format($data['day']['timestamp'], 'custom', 'd');
      $data['day']['month'] = $this->dateFormatter->format($data['day']['timestamp'], 'custom', 'n');
      $data['day']['m'] = $this->dateFormatter->format($data['day']['timestamp'], 'custom', 'm');
      $data['day']['year'] = $this->dateFormatter->format($data['day']['timestamp'], 'custom', 'Y');

      // ISO 8601.
      $data['day']['today']['ISO8601'] = $data['day']['year'] . "-" . $data['day']['m'] . "-" . $data['day']['d'];
      $data['day']['day_of_elaboration']['ISO8601'] = $this->dateFormatter->format(time(), 'custom', 'Y-m-d');
      $data['day']['daybefore']['ISO8601'] = date('Y-m-d', strtotime("-1 day", strtotime($data['day']['today']['ISO8601'])));

      $elab_day = "";
      if ($data['day']['today']['ISO8601'] == $data['day']['day_of_elaboration']['ISO8601']) {
        $elab_day = "<span class='elab-day'>*</span>";
      }

      // Produce columns.
      $columns = [];

      // Column 1 = day.
      $columns[] = Markup::create('<strong>' . $data['day']['formatted'] . $elab_day . '</strong>');
      // Produce a column per unit.
      if (!empty($data['units'])) {
        foreach ($data['units'] as $unit_id => $unit) {
          $data['unit']['node'] = $unit;
          $data['unit']['nid'] = $unit_id;
          $data['unit']['bid'] = $unit->get("field_availability_daily")->target_id;
          $data = $this->cellContent($data);
          $columns[] = $this->renderer->render($data['cellcontent']);
        }
        $data['rows'][] = $columns;
      }
    }
    return $data['rows'];
  }

  /**
   * Get time range for current report.
   */
  public function requestedRange() {
    $data = [];
    $data['session'] = $this->getSession();
    $data['tmp']['a'] = $data['session']->get('beehotel_requested_range');
    $data['tmp']['b'] = $this->defaultTimeRange();
    $data['range'] = $data['session']->get('beehotel_requested_range') ?? $this->defaultTimeRange();
    return $data['range'];
  }

  /**
   * Get rows number for current report.
   *
   * @return array
   *   Data array is enriched with rows number.
   */
  private function rowsNumber() {
    $data = [];
    $data['range'] = $this->RequestedRange();
    $data['ranges'] = $this->beehotelVertical->timeRanges();
    $data['rows'] = $data['ranges'][$data['range']]['rows'];
    return $data['rows'];
  }

  /**
   * Produce cell content(html + data)
   */
  private function cellContent($data) {
    $data = $this->eventUtil->typeofOccupacy($data);
    $data['cellcontent'] = $this->verticalTableTrTd($data, $options = []);
    return $data;
  }

  /**
   * Collect order data.
   */
  private function getOrderData($data, $options) {

    $tmp = [];
    $tmp['day'] = $options['day'];
    $tmp['unit'] = $data['unit']['bid'];
    $tmp['currentday_order_id'] = $this->getCurrentDayOrderId($data);
    $tmp['daybefore_order_id'] = $this->getDayBeforeOrderId($data, $tmp['unit']);
    $order = [];
    $order['total_price__number'] = 0;

    if (
      !empty($data['occupancy'][$tmp['day']][$tmp['unit']])
      && !empty($data['occupancy'][$tmp['day']][$tmp['unit']]['order'])
    ) {

      $order['order_id'] = $data['occupancy'][$tmp['day']][$tmp['unit']]['order']->order_id;
      $order['mail'] = $data['occupancy'][$tmp['day']][$tmp['unit']]['order']->mail;
      $order['total'] = $data['occupancy'][$tmp['day']][$tmp['unit']]['order']->total_price__number;
      $order['order_number'] = $data['occupancy'][$tmp['day']][$tmp['unit']]['order']->order_number;
      $order['order_balance'] = $this->getOrderBalance($order['order_id'], FALSE);

      // No text for same order on following days.
      $order['hidden_text'] = FALSE;

      if (isset($data['occupancy'][$tmp['day']][$tmp['unit']]['order'])) {
        if ($data['occupancy'][$tmp['day']][$tmp['unit']]['order']->object->billing_profile->entity) {
          $address = $data['occupancy'][$tmp['day']][$tmp['unit']]['order']->object->billing_profile->entity->address->getValue()[0];
          $order['name'] = $address['given_name'] ?? '';
          $order['surname'] = $address['surname'] ?? '';
        }
      }

      if ($tmp['currentday_order_id'] != $tmp['daybefore_order_id']) {
        $order['hidden_text'] = TRUE;
      }
      return $order;
    }
  }

  /**
   * Get the balance.
   */
  private function getOrderBalance($order_id, $formatted) {

    $order = $this->entityTypeManager
      ->getStorage('commerce_order')
      ->loadByProperties(['order_id' => $order_id]);

    $balance = $order[$order_id]->getBalance();

    if ($formatted == FALSE) {
      return $balance->getNumber();
    }
    else {
      return $this->currencyFormatter->format($balance->getNumber(), $balance->getCurrencyCode());
    }
  }

  /**
   * Theme cell content.
   */
  private function verticalTableTrTd($data, $options) {

    // A.
    $a_attributes = $this->verticalTableTrTdAttributes($data, $options = ['div' => "a"]);
    $a_class = implode(" ", $a_attributes['class']);
    $a_content = $this->verticalOrderItem($data, $options = ['div' => "a"]);

    // Spacer.
    $spacer_attributes = $this->verticalTableTrTdAttributes($data, $options = ['div' => "spacer"]);
    $spacer_class = implode(" ", $spacer_attributes['class']);
    $spacer_content = "";

    // B.
    $b_attributes = $this->verticalTableTrTdAttributes($data, $options = ['div' => "b"]);
    $b_class = implode(" ", $b_attributes['class']);
    $b_content = $this->verticalTrTdOrderOrState($data, $options = ['div' => "b"]);

    $output = [
      '#theme' => 'vertical_table_tr_td',
      '#a_class' => $a_class,
      '#a_content' => $a_content,
      '#b_class' => $b_class,
      '#b_content' => $b_content,
    ];

    // what's this?
    if (isset($a_content['#order_id']) && (int) $a_content['#order_id'] > 0) {
      $output['#spacer_class'] = $spacer_class;
    }

    return $output;
  }

  /**
   * What to expose in that cell.
   */
  private function verticalTrTdOrderOrState($data, $options = ['div' => "b"]) {
    $content = $this->verticalOrderItem($data, $options = ['div' => "b"]);
    return $content;
  }

  /**
   * Expose that cell state.
   */
  private function verticalTableTrTdState($data, $options) {

    $state = [];
    $state['id'] = $this->eventUtil->getNightState($data);
    if (isset($state['id'])) {
      $state['details'] = $this->eventUtil->getEventState($state['id']);
      $state['label'] = $state['details']->calendar_label;
      $state['color'] = $state['details']->color;
      $state['blocking'] = "";

      if (date('Y-m-d', time()) > $data['day']['today']['ISO8601']) {
        $state['blocking'] = "past";
      }
      elseif ($state['details']->blocking == 1) {
        $state['blocking'] = "blocking";
      }

      $state['extra'] = [
        "#markup" => "<p><a class='use-ajax' data-dialog-type='modal' href='/admin/beehotel/vertical/editstate'>Edit</a></p>",
      ];

      $output = [
        '#theme' => 'event_state_box',
        '#blocking' => $state['blocking'],
        '#color' => $state['color'],
        '#label' => $state['label'],
        '#extra' => $state['extra'],
      ];
      return $output;
    }
  }

  /**
   * Extra info about cell content (maybe events).
   */
  private function verticalTrTdExtra($data, $options) {
    $output = NULL;
    return $output;
  }

  /**
   * Build up order info.
   *
   *   Returns null  when no reservation is fund.
   */
  private function verticalOrderItem($data, $opt) {

    $order = [];

    $tmp = [];
    $tmp['unit'] = $data['unit']['bid'];
    $tmp['currentday_order_id'] = $this->getCurrentDayOrderId($data);
    $tmp['daybefore_order_id'] = $this->getDayBeforeOrderId($data, $tmp['unit']);

    if ($opt['div'] == "a") {
      $options = [
        'day' => $data['day']['daybefore']['ISO8601'],
        'div' => $opt['div'],
      ];
    }
    elseif ($opt['div'] == "b") {
      $options = [
        'day' => $data['day']['today']['ISO8601'],
        'div' => $opt['div'],
      ];
    }

    $order = $this->getOrderData($data, $options);

    $mail = $order['mail'] ?? '';
    $total = $order['total_price__number'] ?? 0;

    if ($order) {
      if ($order['order_balance']) {
        $this->messenger()->addWarning($this->t('IMPORTANT: You have money to collect'));
      }

      $items = $this->entityTypeManager->getStorage('commerce_payment')->loadByProperties(['order_id' => $order['order_id']]);

      $payments = [];
      foreach ($items as $item) {
        if ($item->get('state')->value == "completed") {
          $payments[] = [
            'amount' => number_format($item->get('amount')->number, 2, ',', ' '),
            'payment_id' => $item->get('payment_id')->value,
            'date' => $this->dateFormatter->format($item->get('completed')->value, 'custom', 'd/m/Y'),
          ];
        }
      }

      // @todo add extra info/notes
      $extra = $this->verticalTrTdExtra($data, $options);

      return [
        '#theme' => 'vertical_order_item',
        '#balance' => $order['order_balance'],
        '#extra' => $extra,
        '#hidden_text' => $order['hidden_text'],
        '#mail' => Unicode::truncate($mail, 8, FALSE, TRUE) ,
        '#name' => $order['name'] ?? '',
        '#surname' => $order['surname'] ?? '',
        '#order_id' => $order['order_id'] ?? '' ,
        '#order_number' => $order['order_number'] ?? '',
        '#payments' => $payments ?? '',
        '#total' => number_format($total),
      ];
    }
  }

  /**
   * Build up atributes for table cell.
   */
  private function verticalTableTrTdAttributes($data, $options) {

    $attributes = ['class' => []];
    $tmp['unit'] = $data['unit']['bid'];

    $tmp['currentday_order_id'] = $this->getCurrentDayOrderId($data);
    $tmp['daybefore_order_id'] = $this->getDayBeforeOrderId($data, $tmp['unit']);

    if (empty($data['occupancy'][$data['day']['daybefore']['ISO8601']])) {
      $data['first_row_of_the_table'] = TRUE;
    }

    // A.
    if ($options['div'] == "a") {

      // A1. Same reservation today and yestarday falls  into A2/A3.
      // A2. Guest is checking out today.
      if (!empty($tmp['daybefore_order_id']) && $tmp['currentday_order_id'] != $tmp['daybefore_order_id']) {
        $attributes['class'][] = 'reservation';
        $attributes['class'][] = 'checkout';
      }

      // A3. Today's reservation is the same as yestarday.
      if (!empty($tmp['daybefore_order_id']) && $tmp['currentday_order_id'] == $tmp['daybefore_order_id']) {
        $attributes['class'][] = 'reservation';
        $attributes['class'][] = 'same-as-daybefore';
      }

      // @todo A4. Show Checkout on the first day in table.
    }

    // Spacer.
    if ($options['div'] == "spacer") {
      /*
       * Spacer 1. We have some reservation today and today reservation
       * is not checking in.
       */
      if (!empty($data['occupancy']['current']) &&
          !empty($data['occupancy']['current']['order'])  &&
          $data['occupancy']['current']['order']->checkin != $data['day']['today']['ISO8601'] &&
          $data['occupancy']['current']['order']->checkout != $data['day']['today']['ISO8601']) {
        $attributes['class'][] = 'occupied';
      }

      if (isset($data['occupancy']['current']['event']['id'])) {
        $attributes['class'][] = 'no-display';
      }

    }

    // B.
    if ($options['div'] == "b") {

      // B1. We have a reservation today.
      if (!empty($data['occupancy']['current']) && !empty($data['occupancy']['current']['order'])) {

        $attributes['class'][] = 'reservation';

        // B1.1. Today reservation is checkin today.
        if ($data['occupancy']['current']['order']->checkin == $data['day']['today']['ISO8601']) {
          $attributes['class'][] = 'checkin';
        }
        $content = $this->verticalOrderItem($data, $options = ['div' => "b"]);
      }

    }

    return $attributes;
  }

  /**
   * Add order id to data.
   */
  private function getCurrentDayOrderId($data) {

    if (!empty($data)) {
      if (!empty($data['occupancy'])) {
        if (!empty($data['day']['today']['ISO8601'])) {
          if (!empty($data['occupancy'][$data['day']['today']['ISO8601']])) {
            if (!empty($data['unit']['bid'])) {
              if (!empty($data['occupancy'][$data['day']['today']['ISO8601']][$data['unit']['bid']]['order'])) {
                if (!empty($data['occupancy'][$data['day']['today']['ISO8601']][$data['unit']['bid']]['order'])) {
                  return $data['occupancy'][$data['day']['today']['ISO8601']][$data['unit']['bid']]['order']->order_id;
                }
              }
            }
          }
        }
      }
    }
  }

  /**
   * Get Day before.
   */
  private function getDayBeforeOrderId($data) {
    if (!empty($data)) {
      if (!empty($data['occupancy'])) {
        if (!empty($data['day']['daybefore']['ISO8601'])) {
          if (!empty($data['occupancy'][$data['day']['daybefore']['ISO8601']])) {
            if (!empty($data['unit']['bid'])) {
              if (!empty($data['occupancy'][$data['day']['daybefore']['ISO8601']][$data['unit']['bid']]['order'])) {
                $res = $data['occupancy'][$data['day']['daybefore']['ISO8601']][$data['unit']['bid']]['order']->order_id;
                return $res;
              }
            }
          }
        }
      }
    }
  }

  /**
   * TimeMachine data to step back in time.
   */
  public function page() {
    $result = $this->result($data = []);
    return [
      '#type' => 'markup',
      '#markup' => $result,
    ];
  }

  /**
   * Display a form.
   */
  private function displayTimeRangeForm() {
    $data = [];
    $data['class'] = "\Drupal\beehotel_vertical\Form\TimeRangeForm";
    $form = $this->formBuilder->getForm($data['class']);
    $htmlForm = $this->renderer->render($form);
    return $htmlForm;
  }

  /**
   * Give a default value.
   */
  private function defaultTimeRange() {
    return "cw";
  }

  /**
   * Callback for opening the modal form.
   */
  public function editState() {

    $response = new AjaxResponse();
    // Get the modal form using the form builder.
    $modal_form = $this->formBuilder->getForm('Drupal\beehotel_vertical\Form\BeeHotelVerticalEditEvent');
    // Add an AJAX command to open a modal dialog with the form as the content.
    $response->addCommand(new OpenModalDialogCommand('Edit Event', $modal_form, ['width' => '555']));
    return $response;

  }

  /**
   * Get a fresh session object.
   *
   * @return \Symfony\Component\HttpFoundation\Session\SessionInterface
   *   A session object.
   */
  protected function getSession() {
    return $this->requestStack->getCurrentRequest()->getSession();
  }

}
