<?php

namespace Drupal\bee_hotel\Form;

use CommerceGuys\Intl\Formatter\CurrencyFormatterInterface;
use Drupal\bee_hotel\Util\BeeHotelUnit;
use Drupal\commerce\Context;
use Drupal\commerce_price\Resolver\ChainPriceResolverInterface;
use Drupal\commerce_order\PriceCalculatorInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\node\NodeInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Session\Session;

/**
 * Search form for available beehotel units.
 */
class UnitsSearch extends FormBase {

  /**
   * The chain price resolver.
   *
   * @var \Drupal\commerce_price\Resolver\ChainPriceResolverInterface
   */
  protected $chainPriceResolver;

  /**
   * The currency formatter.
   *
   * @var \CommerceGuys\Intl\Formatter\CurrencyFormatterInterface
   */
  protected $currencyFormatter;

  /**
   * The Bee Hotel Unit utility.
   *
   * @var \Drupal\bee_hotel\Util\BeeHotelUnit
   */
  protected $beeHotelUnit;

  /**
   * The price calculator.
   *
   * @var \Drupal\commerce_order\PriceCalculatorInterface
   */
  protected $priceCalculator;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The session.
   *
   * @var \Symfony\Component\HttpFoundation\Session\Session
   */
  protected $session;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a new UnitsSeach object.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   User account.
   * @param \Drupal\commerce_price\Resolver\ChainPriceResolverInterface $chain_price_resolver
   *   Chain price resolver.
   * @param \CommerceGuys\Intl\Formatter\CurrencyFormatterInterface $currency_formatter
   *   The currency formatter.
   * @param \Drupal\bee_hotel\Util\BeeHotelUnit $bee_hotel_unit
   *   The BeeHotelUnit utility.
   * @param \Drupal\commerce_order\PriceCalculatorInterface $price_calculator
   *   The price calculator.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Symfony\Component\HttpFoundation\Session\Session $session
   *   The session.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   */
  public function __construct(AccountInterface $account, ChainPriceResolverInterface $chain_price_resolver, CurrencyFormatterInterface $currency_formatter, BeeHotelUnit $bee_hotel_unit, PriceCalculatorInterface $price_calculator, EntityTypeManagerInterface $entity_type_manager, Session $session, RendererInterface $renderer) {
    $this->account = $account;
    $this->chainPriceResolver = $chain_price_resolver;
    $this->currencyFormatter = $currency_formatter;
    $this->beeHotelUnit = $bee_hotel_unit;
    $this->priceCalculator = $price_calculator;
    $this->entityTypeManager = $entity_type_manager;
    $this->session = $session;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      // Load required services to construct this class.
      $container->get('current_user'),
      $container->get('commerce_price.chain_price_resolver'),
      $container->get('commerce_price.currency_formatter'),
      $container->get('bee_hotel.util.beehotelunit'),
      $container->get('commerce_order.price_calculator'),
      $container->get('entity_type.manager'),
      $container->get('session'),
      $container->get('renderer'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'beehotel_units_search';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, NodeInterface $node = NULL) {

    $config = $this->config('beehotel.settings');

    if ($config->get("beehotel.off_value")) {
      $this->messenger()->addWarning($config->get("beehotel.off_text"));
      return;
    }

    // Clean booking related data in session.
    $this->session->set('beehotel_data', []);

    $today = new \DateTime();

    $tomorrow = clone($today);
    $tomorrow->modify('+1 day');

    $one_hour_later = clone($today);
    $one_hour_later->modify('+1 hour');

    $form['#prefix'] = "<div id='units-search-form-container'>";

    // Dates.
    $form['dates'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Check in > Check out'),
      '#default_value' => date("j M Y", strtotime("+1 day")) . " - " . date("j M Y", strtotime("+3 day")),
      '#required' => TRUE,
      '#weight' => -1000,
    ];

    // People (occupants).
    $max_occupancy = $this->beeHotelUnit->maxOccupancy($node = NULL);
    $options = [];
    for ($o = 1; $o <= $max_occupancy; $o++) {
      $options[$o] = $o . " " . $this->t("guests");
    }

    $form['guests'] = [
      '#type' => 'select',
      '#title' => $this->t('Guests'),
      '#options' => $options,
      '#required' => TRUE,
    ];

    $form['submit'] = [
      '#type' => 'button',
      '#value' => $this->t('Search'),
    ];

    $form['#suffix'] = "</div>";

    if ($form_state->getTriggeringElement()) {
      $items = $this->produceResult($form_state);
      $form['#suffix'] .= "<div class='label'>" . $this->t("Results") . "</div><div class='result units'>" . $this->renderer->render($items) . "</div>";
    }

    $form['#attached']['library'][] = 'bee_hotel/beehotel-litepicker';
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $data['values'] = $form_state->getValues();
    $data = $this->beeHotelUnit->getAvailableUnits($data);
    // Basic date validation.
    if (empty($data['available_units'])) {
      $form_state->setErrorByName('checkout', $this->t('No availability, sorry...'));
      return;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $form_state->setRebuild(TRUE);
  }

  /**
   * Produce results.
   *
   * Produce results from submitted values.
   */
  private function produceResult($form_state) {
    $data = [];
    $data['values'] = $form_state->getValues();

    $data = $this->beeHotelUnit->getAvailableUnits($data);

    foreach ($data['available_units'] as $unit_id) {

      $beeHotelUnit = $this->beeHotelUnit->getUnitFromBid($unit_id);

      if (isset($beeHotelUnit)) {

        // Unit disabled.
        if ($beeHotelUnit['node']->get("status")->value != 1) {
          continue;
        }

        // Unit not accepting reservations.
        if ($beeHotelUnit['node']->get("field_accept_reservations")->value != 1) {
          continue;
        }

        $variation = $this->beeHotelUnit->GetCurrentVariation($beeHotelUnit, $data);

        if (isset($variation)) {

          $data['v_param'] = $data['norm']['dates_from_search_form']['checkin']['Y-m-d'] .
            "-" .
            $data['norm']['dates_from_search_form']['checkout']['Y-m-d'] .
            "-" .
            $data['values']['guests'];

          $data['destination']['title'] = Link::createFromRoute($beeHotelUnit['node']->getTitle(), 'entity.node.canonical', [
            'node' => $beeHotelUnit['node']->Id(),
            'v' => $data['v_param'],
          ]);

          $data['destination']['text'] = Link::createFromRoute($this->t('Book now'), 'entity.node.canonical', [
            'node' => $beeHotelUnit['node']->Id(),
            'v' => $data['v_param'],
          ]);

          $data['destination']['img'] = Link::createFromRoute($beeHotelUnit['cover_image']['markup'], 'entity.node.canonical', [
            'node' => $beeHotelUnit['node']->Id(),
            'v' => $data['v_param'],
          ]);

          $context = new Context($this->account,
                                 $this->entityTypeManager->getStorage('commerce_store')->load($beeHotelUnit['store']->Id())
          );

          $purchasable_entity = $variation;
          $data['addtocart']['caller'] = __CLASS__ . "::" . __FUNCTION__;
          $this->beeHotelUnit->storeInSession($data);

          $price = $this->priceCalculator->calculate($variation, 1, $context);
          $calculatedprice = $price->getCalculatedPrice();

          $formattedprice = $this->currencyFormatter->format($calculatedprice->getNumber(), $calculatedprice->getCurrencyCode(), [
            'locale' => $beeHotelUnit['store']->get("langcode")->value,
            'minimum_fraction_digits' => 2,
            'maximum_fraction_digits' => 2,
            'currency_display' => 'none',
          ]);

          $items[] = [
            '#theme' => 'bee_hotel_s_unit',
            '#currency' => $calculatedprice->getCurrencyCode(),
            '#img' => $data['destination']['img'],
            '#nights' => $data['norm']['dates_from_search_form']['days'],
            '#price' => $formattedprice,
            '#title' => $data['destination']['title'],
            '#store' => $beeHotelUnit['store']->get('name')->value,
            '#product_id' => $beeHotelUnit['pid'],
            '#variation_id' => (int) $purchasable_entity->id(),
            '#destination' => $data['destination']['text'],
          ];

          // Optional field for marketing pourposes.
          if ($beeHotelUnit['node']->hasField('field_slogan')) {
            $items['#description'] = $beeHotelUnit['node']->get("field_slogan")->value;
          }

        }
      }
    }
    return $items;
  }

}
