<?php

namespace Drupal\bee_hotel\Resolvers;

use Drupal\beehotel_pricealterator\Alter;
use Drupal\beehotel_pricealterator\PreAlter;
use Drupal\bee_hotel\Util\BeeHotelUnit;
use Drupal\bee_hotel\Util\Dates;
use Drupal\commerce\Context;
use Drupal\commerce\PurchasableEntityInterface;
use Drupal\commerce_price\Price;
use Drupal\commerce_price\Resolver\PriceResolverInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Interacts with Commerce price system.
 *
 * @package Drupal\bee_hotel\Resolvers
 */
class SalepriceResolver implements PriceResolverInterface {

  /**
   * The dates Utilities.
   *
   * @var \Drupal\bee_hotel\Util\Dates
   */
  protected $datesUtil;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The bee hotel unit.
   *
   * @var \Drupal\bee_hotel\Util\BeeHotelUnit
   */
  private $beehotelunit;

  /**
   * The Alter manager.
   *
   * @var \Drupal\beehotel_pricealterator\Alter
   */
  private $alterManager;

  /**
   * The Prealter manager.
   *
   * @var \Drupal\beehotel_pricealterator\Prealter
   */
  private $preAlterManager;

  /**
   * Constructs the object.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\bee_hotel\Util\Dates $bee_hotel_dates
   *   The BeeHotel Dates Utility.
   * @param \Drupal\bee_hotel\Util\BeeHotelUnit $bee_hotel_unit
   *   The BeeHotel Unit Utility.
   * @param \Drupal\beehotel_pricealterator\Prealter $pre_alter_manager
   *   The BeeHotel Unit Utility.
   * @param \Drupal\beehotel_pricealterator\Alter $alter_manager
   *   The BeeHotel Unit Utility.
   */
  public function __construct(RequestStack $request_stack, Dates $bee_hotel_dates, BeeHotelUnit $bee_hotel_unit, Prealter $pre_alter_manager, Alter $alter_manager) {
    $this->requestStack = $request_stack;
    $this->datesUtil = $bee_hotel_dates;
    $this->beehotelunit = $bee_hotel_unit;
    $this->preAlterManager = $pre_alter_manager;
    $this->alterManager = $alter_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack'),
      $container->get('bee_hotel.util.dates'),
      $container->get('bee_hotel.util.beehotelunit'),
      $container->get('beehotel_pricealterator.prealter'),
      $container->get('beehotel_pricealterator.alter'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function resolve(PurchasableEntityInterface $entity, $quantity, Context $context) {

    if ($entity->bundle() != 'bee') {
      return;
    }

    $data = [];

    $data['stores'] = $entity->getProduct()->getStores();
    $data['store'] = reset($data['stores']);
    $data['currency_code'] = $data['store']->get('default_currency')->getValue()[0]['target_id'];
    $data['order'] = $this->requestStack->getCurrentRequest()->query->get("commerce_order");

    if (!empty($this->requestStack->getCurrentRequest()->request->get("dates"))) {
      $data['values']['dates'] = $this->requestStack->getCurrentRequest()->request->get("dates");
      $this->datesUtil->normaliseDatesFromSearchForm($data);
      $this->datesUtil->easter($data);
      $dates = $data['norm']['dates_from_search_form'];

      if (
         isset($dates['checkin']) &&
         isset($dates['checkout']) &&
         $this->requestStack->getCurrentRequest()->request->get("guests")) {

        if ($this->requestStack->getCurrentRequest()->request->get("form_id") == "bee_hotel_book_this_unit_form") {
          $this->fromSingleForm($data, $context, $entity);
        }
        elseif ($this->requestStack->getCurrentRequest()->request->get("form_id") == "beehotel_units_search") {
          $this->fromUnitsSearchForm($data, $context, $entity);
        }
        else {
          return;
        }

        $data['amount'] = bee_hotel_number_format($data['amount']);
        $price = new Price($data['amount'], $data['currency_code']);
        return $price;
      }
    }
  }

  /**
   * Get useful data.
   */
  private function fromSingleForm(&$data, $context, $entity) {
    $data['adults'] = (int) $this->requestStack->getCurrentRequest()->request->get("guests");
    $data['context'] = $context;
    $data['date'] = time();
    $data['entity'] = $entity;
    $data['node'] = $this->beehotelunit->getVariationNode($entity);
    $data['nid'] = $data['node']->Id();
    $data['season'] = $this->preAlterManager->season($data);
    $data = $this->alterManager->alter($data);
  }

  /**
   * Get useful data.
   */
  private function fromUnitsSearchForm(&$data, $context, $entity) {
    $data['adults'] = (int) $this->requestStack->getCurrentRequest()->request->get("guests");
    $data['context'] = $context;
    $data['date'] = time();
    $data['entity'] = $entity;
    $data['node'] = $this->beehotelunit->getVariationNode($entity);
    $data['nid'] = $data['node']->Id();

    $data['season'] = $this->preAlterManager->season($data);
    $data = $this->alterManager->alter($data);
  }

}
