<?php

namespace Drupal\beehotel_pricealterator;

use Drupal\bee_hotel\Util\BeeHotelCommerce;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class Alter in charge of the price alteration.
 *
 * @package Drupal\beehotel_pricealterator\Services
 */
class Alter {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The tempstore factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $tempStoreFactory;

  /**
   * The preparing Class.
   *
   * @var \Drupal\beehotel_pricealterator\PreAlter
   */
  protected $preAlter;

  /**
   * The plugin manager.
   *
   * @var \Drupal\beehotel_pricealterator\PriceAlteratorPluginManager
   */
  protected $priceAlteratorPluginManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The BeeHotel commerce Util.
   *
   * @var \Drupal\bee_hotel\Util\BeeHotelCommerce
   */
  protected $beehotelCommerce;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Alter constructor.
   *
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The tempstore factory.
   * @param \Drupal\beehotel_pricealterator\PreAlter $pre_alter
   *   The tempstore factory.
   * @param \Drupal\beehotel_pricealterator\PriceAlteratorPluginManager $price_alterator_plugin_manager
   *   The tempstore factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\bee_hotel\Util\BeeHotelCommerce $beehotel_commerce
   *   BeeHotel Commerce Utils.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   */
  public function __construct(
    AccountInterface $currentUser,
    PrivateTempStoreFactory $temp_store_factory,
    PreAlter $pre_alter,
    PriceAlteratorPluginManager $price_alterator_plugin_manager,
    EntityTypeManagerInterface $entity_type_manager,
    BeeHotelCommerce $beehotel_commerce,
    RendererInterface $renderer,
  ) {
    $this->currentUser = $currentUser;
    $this->tempStoreFactory = $temp_store_factory;
    $this->preAlter = $pre_alter;
    $this->priceAlteratorPluginManager = $price_alterator_plugin_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->beehotelCommerce = $beehotel_commerce;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_user'),
      $container->get('tempstore.private'),
      $container->get('beehotel_pricealterator.prealter'),
      $container->get('plugin.manager.beehotel.pricealterator'),
      $container->get('entity_type.manager'),
      $container->get('bee_hotel.util.beehotelcommerce'),
      $container->get('renderer'),
    );
  }

  /**
   * Contains the alteration flow.
   */
  public function alter($data) {

    $data['tempstore'] = $tempstore = $this->tempStoreFactory->get('beehotel_pricealterator');

    // 1. Get content.
    // 1a. Base Table with base price.
    $data['basetable'] = $basetable = $this->preAlter->baseTable($data);

    // 1b. Available Alterators.
    $alterators = $this->priceAlteratorPluginManager->alterators($data);

    // 1c. Annotation status Alterators.
    $alterators = $this->checkStatus($alterators);

    // 1c. UI enabled  Alterators.
    $data['alterators'] = $alterators = $this->checkEnabled($alterators);

    $tempstore->set('beehotel_pricealterator_log', $alterators);

    /* 2. Loop per night.
     * Drupal Commerce calls Resolver once per order item (not once per night)
     * We loop nights here, providing an average price
     * to be multiplied by items (nights).
     * @todo keep track of every night price into the $data array.
     */
    for ($n = 0; $n < $data['norm']['dates_from_search_form']['days']; $n++) {
      $data['tmp'] = NULL;
      $data['tmp']['night_timestamp'] =
        $data['norm']['dates_from_search_form']['checkin']['timestamp'] + (24 * 3600 * $n);
      $data['tmp']['night_date'] = date('Y-m-d', $data['tmp']['night_timestamp']);
      foreach ($alterators as $a) {
        $alterator = new $a['class']([], [], [], $this->beehotelCommerce, $this->renderer);
        $data = $alterator->alter($data, $basetable);
        if (isset($data['tmp']['price'])) {
          $price[$data['tmp']['night_timestamp']] = $data['tmp']['price'];
        }
      }
    }
    $data['price'] = $price = array_filter($price);

    if (count($price)) {
      $average = array_sum($price) / count($price);
    }

    $data['amount'] = $average;
    return $data;
  }

  /**
   * Check annotation status.
   */
  private function checkStatus($alterators) {
    $a = [];
    foreach ($alterators as $item) {
      if ($item['status'] == 1) {
        $a[] = $item;
      }
    }
    return $a;
  }

  /**
   * Check UI enabled.
   */
  private function checkEnabled($alterators) {

    $a = [];
    foreach ($alterators as $item) {
      if ($item['enabled'] == 1) {
        $a[] = $item;
      }
    }

    return $a;
  }

  /**
   * How to send this output to the debug block?
   *
   * @todo implement the feature.
   */
  public function beehotelLog($data, $context) {}

}
