<?php

namespace Drupal\bee_hotel\Util;

use Drupal\commerce_product\Entity\ProductAttribute;
use Drupal\commerce_product\ProductAttributeFieldManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Utilities for BeeHotel.
 */
class BeeHotel implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * The attribute field manager.
   *
   * @var \Drupal\commerce_product\ProductAttributeFieldManagerInterface
   */
  protected $attributeFieldManager;

  /**
   * The bee hotel unit.
   *
   * @var \Drupal\bee_hotel\Util\BeeHotelUnit
   */
  private $beehotelunit;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new ProductAttributeForm object.
   *
   * @param \Drupal\commerce_product\ProductAttributeFieldManagerInterface $attribute_field_manager
   *   The attribute field manager.
   * @param \Drupal\bee_hotel\Util\BeeHotelUnit $bee_hotel_unit
   *   The BeeHotel Unit Utility.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   */
  public function __construct(
    ProductAttributeFieldManagerInterface $attribute_field_manager,
    BeeHotelUnit $bee_hotel_unit,
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    MessengerInterface $messenger) {
    $this->attributeFieldManager = $attribute_field_manager;
    $this->beehotelunit = $bee_hotel_unit;
    $this->config = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('commerce_product.attribute_field_manager'),
      $container->get('bee_hotel.util.beehotelunit'),
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('messenger')
    );
  }

  /**
   * Node SETUP mode.
   */
  public function checkBeeHotelSetupNode(EntityInterface $node) {

    $this->checkGuestAttribute();

    // Accept reservation.
    $this->beehotelunit->checkUnitAcceptsReservations($node);

    // Correct variations.
    $this->beehotelunit->checkProductVariations($node);

    // Weekly price table.
    $this->beehotelunit->checkBasePriceTable($node);
  }

  /**
   * Guest attribute check.
   */
  public static function checkGuestAttribute() {
    // A. Guests attributes required.
    if (!ProductAttribute::load('guests')) {
      $tmp = t("BEE Hotel requires a commerce  attribute named 'guests', not found in current configuration.") . " ";
      $tmp .= t("<a href='@path'>Fix this</a> now", ['@path' => '/admin/commerce/product-attributes']);
      \Drupal::messenger()->addError(Markup::create($tmp));
    }

    // B. BEE flag as variation type.
    $attribute_field_manager = \Drupal::service('commerce_product.attribute_field_manager');
    $map = $attribute_field_manager->getFieldMap();
    if (!isset($map['bee'])) {
      $link = Link::fromTextAndUrl('Set BEE as "Product variation types"', Url::fromUserInput('/admin/commerce/product-attributes/manage/guests'));
      \Drupal::messenger()->addWarning($link);
    }
  }

  /**
   * Libraries check.
   */
  public static function checkLibraries() {
    $data = [];
    // A. Litepicker library.
    $data['litepicker'] = '/libraries/litepicker/litepicker.js';
    if (!file_exists(DRUPAL_ROOT . $data['litepicker'])) {
      $link = Link::fromTextAndUrl('Litepicker library is missing"', Url::fromUri('https://www.drupal.org/node/3126948/edit'));
      \Drupal::messenger()->addWarning($link);
    }
  }

  /**
   * Check current page is admin.
   */
  public static function isAdmin() {
    if (\Drupal::service('router.admin_context')) {
      return TRUE;
    }
    return FALSE;
  }

  /**
   * Produce results.
   *
   * Produce results from submitted values.
   */
  private function produceResult($values) {
    $data = [];
    $data['values'] = $values;

    $data = $this->beeHotelUnit->getAvailableUnits($data);

    // Get current language code.
    $data['current_language'] = \Drupal::languageManager()->getCurrentLanguage()->getId();

    foreach ($data['available_units'] as $unit_id) {

      $beeHotelUnit = $this->beeHotelUnit->getUnitFromBid($unit_id);

      if (isset($beeHotelUnit)) {

        // Unit disabled.
        if ($beeHotelUnit['node']->get("status")->value != 1) {
          continue;
        }

        // Unit not accepting reservations.
        if ($beeHotelUnit['node']->get("field_accept_reservations")->value != 1) {
          continue;
        }

        // Language.
        if (!$beeHotelUnit['node']->hasTranslation($data['current_language'])) {
          continue;
        }

        // We only want the translated node.
        $beeHotelUnit['translated_node'] = $beeHotelUnit['node']->getTranslation($data['current_language']);
        $beeHotelUnit['node'] = $beeHotelUnit['translated_node'];

        $variation = $this->beeHotelUnit->GetCurrentVariation($beeHotelUnit, $data);

        if (isset($variation)) {

          $data['v_param'] = $data['norm']['dates_from_search_form']['checkin']['Y-m-d'] .
            "-" .
            $data['norm']['dates_from_search_form']['checkout']['Y-m-d'] .
            "-" .
            $data['values']['guests'];

          $data['destination']['title'] = Link::createFromRoute($beeHotelUnit['node']->getTitle(), 'entity.node.canonical', [
            'node' => $beeHotelUnit['node']->Id(),
            'v' => $data['v_param'],
          ]);

          $data['destination']['text'] = Link::createFromRoute($this->t('Book now'), 'entity.node.canonical', [
            'node' => $beeHotelUnit['node']->Id(),
            'v' => $data['v_param'],
          ]);

          $data['destination']['img'] = Link::createFromRoute($beeHotelUnit['cover_image']['markup'], 'entity.node.canonical', [
            'node' => $beeHotelUnit['node']->Id(),
            'v' => $data['v_param'],
          ]);

          $context = new Context($this->account,
                                 $this->entityTypeManager->getStorage('commerce_store')->load($beeHotelUnit['store']->Id())
          );

          $purchasable_entity = $variation;
          $data['addtocart']['caller'] = __CLASS__ . "::" . __FUNCTION__;
          $this->beeHotelUnit->storeInSession($data);

          $price = $this->priceCalculator->calculate($variation, 1, $context);
          $calculatedprice = $price->getCalculatedPrice();

          $formattedprice = $this->currencyFormatter->format($calculatedprice->getNumber(), $calculatedprice->getCurrencyCode(), [
            'locale' => $beeHotelUnit['store']->get("langcode")->value,
            'minimum_fraction_digits' => 2,
            'maximum_fraction_digits' => 2,
            'currency_display' => 'none',
          ]);

          $items[] = [
            '#theme' => 'bee_hotel_s_unit',
            '#currency' => $calculatedprice->getCurrencyCode(),
            '#img' => $data['destination']['img'],
            '#nights' => $data['norm']['dates_from_search_form']['days'],
            '#price' => $formattedprice,
            '#title' => $data['destination']['title'],
            '#store' => $beeHotelUnit['store']->get('name')->value,
            '#product_id' => $beeHotelUnit['pid'],
            '#variation_id' => (int) $purchasable_entity->id(),
            '#destination' => $data['destination']['text'],
          ];

          // Optional field for marketing pourposes.
          if ($beeHotelUnit['node']->hasField('field_slogan')) {
            $items['#description'] = $beeHotelUnit['node']->get("field_slogan")->value;
          }

        }
      }
    }
    return $items;
  }

}
