<?php

namespace Drupal\bef_links_filter\Plugin\better_exposed_filters\filter;

use Drupal\better_exposed_filters\Plugin\better_exposed_filters\filter\Links;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Form\FormStateInterface;

/**
 * Filterable list of facet links.
 *
 * @BetterExposedFiltersFilterWidget(
 *   id = "bef_links_filter",
 *   label = @Translation("Filterable links"),
 * )
 */
class LinksFilter extends Links {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return parent::defaultConfiguration() + [
      'minimum_to_show_text_filter' => 5,
      'initial_shown' => 10,
      'allow_show_all' => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['minimum_to_show_text_filter'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum number of facet links before showing text filter'),
      '#default_value' => $this->configuration['minimum_to_show_text_filter'],
      "#mininum" => 1,
      "#maximum" => 500,
      '#description' => $this->t('If there are only this many links to choose from, do not show the text box for filtering.  Set to one to always show the filter even if only one link to filter (not recommended).'),
    ];

    $form['initial_shown'] = [
      '#type' => 'number',
      '#title' => $this->t('Initial number of links shown'),
      '#default_value' => $this->configuration['initial_shown'],
      '#description' => $this->t('This will be ignored once filtering starts.  Set to zero to not show any links until filtered.  Set to 1,000 to show all links.'),
    ];

    $form['allow_show_all'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow showing all links'),
      '#default_value' => !empty($this->configuration['allow_show_all']),
      '#description' => $this->t('When this is enabled, a "show all" link will be at the bottom of the initially shown links above.  This goes away once filtering starts.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function exposedFormAlter(array &$form, FormStateInterface $form_state): void {
    /** @var \Drupal\views\Plugin\views\filter\FilterPluginBase $filter */
    $filter = $this->handler;
    $field_id = $this->getExposedFilterFieldId();

    parent::exposedFormAlter($form, $form_state);

// This is not a real form field so no reason not to do it in the template
//    $form[$field_id]['bef_links_filter'] = [
//      '#type' => 'textfield',
//      '#title' => $this->t('Filter links'),
//      '#attributes' => ['class' => 'bef-links-filter'],
//    ];

    if (!empty($form[$field_id])) {
      $minimum_to_show_text_filter = $this->configuration['minimum_to_show_text_filter'] ?? 0;
      $link_count = count($form[$field_id]['#options']);
      if ($minimum_to_show_text_filter < $link_count) {
        $initial_shown = $this->configuration['initial_shown'] ?? 1000;
        $allow_show_all = $this->configuration['allow_show_all'] ?? FALSE;
        // If the number of initially shown links is set to the max (or above)
        // do not hide any links; the allow_show_all variable will be ignored.
        if ($initial_shown >= $link_count) {
          $initial_shown = FALSE;
        }
        $form[$field_id]['#theme'] = 'bef_links_filter';
        $form[$field_id]['#attributes']['class'][] = 'bef-links-filter';
        $form[$field_id]['#bef_links_filter_initial_shown'] = $initial_shown;
        $form[$field_id]['#bef_links_filter_allow_show_all'] = $allow_show_all;
        $form['#attached']['library'][] = 'bef_links_filter/filter';
      }
    }
    else {
      // Hack to fix BEF bug.
      unset($form[$field_id . '_collapsible']);
    }
  }

}
