<?php

/**
 * @file
 */


/**
 * Be kind on poor Drupal!
 */
define('BEHAT_ERROR_REPORTING', E_ERROR | E_WARNING | E_PARSE);

use Behat\Behat\Context\ClosuredContextInterface,
    Behat\Behat\Context\TranslatedContextInterface,
    Behat\Behat\Context\BehatContext,
    Behat\Behat\Exception\PendingException,
    Behat\Mink\Exception\ElementNotFoundException,
    Behat\Behat\Context\Step;
use Behat\Gherkin\Node\PyStringNode,
    Behat\Gherkin\Node\TableNode;

use Behat\MinkExtension\Context\MinkContext;
use Behat\Mink\Session;

/**
 * Features context.
 */
class DrupalContext extends MinkContext {

  public static $user_roles = array();

  /**
   * Initializes context.
   * Every scenario gets it's own context object.
   *
   * @param array $parameters
   *   context parameters (set them up through behat.yml)
   */
  public function __construct(array $parameters) {

    $this->parameters = $parameters;

    // Initialize your context here.
    $_SERVER['PHP_SELF'] = basename(__FILE__);
    $_SERVER['REMOTE_ADDR'] = $parameters['site'];
    $_SERVER['HTTP_HOST'] = $parameters['site'];
    $_SERVER['REQUEST_METHOD'] = 'GET';

    if (!defined('DRUPAL_ROOT')) {
      define('DRUPAL_ROOT', getcwd());
    }

    require_once DRUPAL_ROOT . '/includes/bootstrap.inc';
    drupal_bootstrap(DRUPAL_BOOTSTRAP_FULL);
  }

  /**
   * Returns Mink session.
   *
   * @param string|NULL $name
   *   name of the session OR active session will be used
   *
   * @return session
   *   A MINK session
   */
  public function getSession($name = NULL) {
    $session = $this->getMink()->getSession($name);

    // This forces the creation of a test version
    // of the database for use with behat.
    // Since it hooks into Simpletest it must use
    // simpletests naming convention.
    $ua = drupal_generate_test_ua('simpletest65');
    $session->setRequestHeader('User-Agent', $ua);

    return $session;
  }

  /** @BeforeSuite */
  public static function prepareForTheSuite($event) {
    behat_testing_setup();
    $parameters = $event->getContextParameters();

    // If configuration requires the test database to be rebuilt
    // do so here.
    if ($parameters['use_site_clone'] == 1) {
      // Clone the existing site.
      behat_testing_copy_database('simpletest65');
      behat_testing_setup_paths();
    }
    else {
      // Clean up any previous test runs.
      behat_testing_clean_database();
    }

  }

  /** @BeforeFeature */
  public static function beforeFeature($event) {
    $parameters = $event->getContextParameters();

    // Reinstall site.
    if (isset($parameters['rebuild_on_feature']) && !empty($parameters['rebuild_on_feature'])) {
      // Clear out statics.
      self::$user_roles = array();
      // Clear out the db.
      behat_testing_clean_database();
      // Reinstall.
      behat_testing_reinstall($parameters['rebuild_on_feature']);

      if (isset($parameters['rebuild_module']) && !empty($parameters['rebuild_module'])) {
        module_enable(explode(',', $parameters['rebuild_module']));
      }
    }
  }

  /** @BeforeStep */
  public static function prepareForTheStep() {
  }

  /**
   * @Given /^I am using a test database$/
   */
  public function iAmUsingATestDatabase() {
  }


  /**
   * @Given /^I am logged in with the role "([^"]*)"$/
   */
  public function iAmLoggedInWithTheRole($role) {
    if (!isset(self::$user_roles[$role])) {
      $r = user_role_load_by_name($role);

      if (empty($r)) {
        throw new Exception(t('No such role @role', array('@role' => $role)));
      }

      $account = $this->drupalCreateUser($this->randomName(), $role);

      if (empty($account)) {
        throw new Exception(t('Could not create user with role @role', array('@role' => $role)));
      }

      self::$user_roles[$role] = $account->name;
    }

    return array(
      new Step\Given('I am logged in as "' . self::$user_roles[$role] . '"'),
    );
  }

  /**
   * @Given /^I should see a "([^"]*)" field with id "([^"]*)"$/
   */
  public function iShouldSeeAFieldWithId($field_type, $id) {
    $element = $this->assertSession()->elementExists('css', $field_type . '#' . $id);
  }

  /**
   * @Then /^I should see nodes of type "([^"]*)" inside "([^"]*)"$/
   */
  public function iShouldSeeNodesOfTypeInside($type, $selector) {
    $element = $this->assertSession()->elementExists('css', $selector);
    $types = $element->find('css', '.node-' . $type);
    if (NULL === $types) {
      throw new ElementNotFoundException($this->getSession(), 'element', $type, $selector);
    }
    return $types;
  }

  /**
   * @Given /^I go to the "([^"]*)" node form$/
   */
  public function iGoToTheNodeForm($nodeform) {
    list($operation, $type) = explode(" ", $nodeform);
    if ($operation == 'add') {
      return array(
        new Step\Given('I am on "node/add/' . $type . '"'),
      );
    }
  }

  /**
   * @Then /^I should see nodes of type "([^"]*)" inside "([^"]*)" with taxonomy "([^"]*)" matching "([^"]*)"$/
   */
  public function iShouldSeeNodesOfTypeInsideWithTaxonomyMatching($type, $selector, $node_property, $matching) {
    $node = $this->iShouldSeeNodesOfTypeInside($type, $selector);
    $node = explode('-', ($node->getAttribute('id')));
    $nid = $node[1];
    $n = node_load($nid);
    $terms = field_view_field('node', $n, $node_property);
    foreach ($terms as $id => $term) {
      if (is_numeric($id)) {
        if ($term['#title'] == $matching) {
          return TRUE;
        }
      }
    }
    throw new ElementNotFoundException($this->getSession(), 'element', $type, $selector);
  }

  /**
   * @Given /^I am on the view "([^"]*)" with the arguments "([^"]*)"$/
   */
  public function iAmOnTheViewWithTheArgument($view, $args) {
    list($view_name, $display) = explode('|', $view);
    $v = views_get_view($view_name);
    $url = $v->display[$display]->display_options['path'];

    if ($url) {
      return new Step\Given('I am on "/' . $url . '/' . $args . '"');
    }
  }

  /**
   * @Given /^a node of type "([^"]*)" with the title "([^"]*)" and language "([^"]*)" exists$/
   */
  public function aNodeOfTypeWithTheTitleAndLanguageExists($content_type, $title, $language) {
    $account = user_load(1);
    $query = new EntityFieldQuery();
    $result = $query->entityCondition('entity_type', 'node')
      ->entityCondition('bundle', $content_type)
      ->propertyCondition('title', $title)
      ->propertyCondition('language', $language)
      ->addMetaData('account', $account)
      ->execute();

    $nodes = isset($result['node']) ? entity_load('node', array_keys($result['node'])) : array();

    if (empty($nodes)) {
      $node = new stdClass();
      $node->type = $content_type;
      $node->title = $title;
      $node->language = $language;
      node_save($node);
    }
  }

  /**
   * @Given /^a node of type "([^"]*)" with the title "([^"]*)" exists$/
   */
  public function aNodeOfTypeWithTheTitleExists($content_type, $title) {
    return $this->aNodeOfTypeWithTheTitleAndLanguageExists($content_type, $title, 'und');
  }

  /**
   * @Given /^cron has run$/
   */
  public function cronHasRun() {
    return new Step\Given('I am on "/cron.php?cron_key=' . variable_get('cron_key', 'drupal') . '"');
  }

  /**
   * @Given /^I am not logged in$/
   */
  public function iAmNotLoggedIn() {
    $this->getSession()->reset();
  }

  /**
   * @Given /^I should see a "([^"]*)" field with id "([^"]*)" and value "([^"]*)"$/
   */
  public function iShouldSeeAFieldWithIdAndValue($field_type, $id, $value) {
    $page = $this->getSession()->getPage();
    $field = $page->find('css', $field_type . '#' . $id);
    if ($field->getValue() != $value) {
      throw new Exception("$field_type with id $id does not equal $value. Found '" . $field->getValue() . "' instead");
    }
  }

  /**
   * @Given /^there should be a form error on the element with id "([^"]*)"$/
   */
  public function thereShouldBeAFormErrorOnTheElementWithId($id) {
    $page = $this->getSession()->getPage();
    $field = $page->find('css', '#' . $id);
    if (empty($field)) {
      throw new Exception('No element with id ' . $id);
    }

    if (!$this->elementHasClass($field, 'error')) {
      throw new Exception("The element with id $id has not been flagged as in an error state");
    }
  }

  /**
   * Helper function to determine if an element has a class.
   */
  public function elementHasClass($element, $class) {
    $classes = explode(' ', $element->getAttribute('class'));
    return in_array($class, $classes);
  }

  /**
   * @Given /^I am logged in as "([^"]*)"$/
   */
  public function iAmLoggedInAs($name) {

    // Load the user fresh.
    $users = user_load_multiple(array(), array('name' => $name), TRUE);

    if (empty($users)) {
      throw new Exception(t('No such user with username "@name"', array('@name' => $name)));
    }

    $user = array_pop($users);

    $session = $this->getSession();
    $link = $this->generateOneTimeLogin($user);
    $session->visit($link);
    $page = $session->getPage();
    $page->find('css', 'input#edit-submit')->press();
  }

  /**
   * Generate a one time login link.
   *
   * @param user $account
   *   The user account you need to login as
   *
   * @return string
   *   URL to login
   */
  public function generateOneTimeLogin($account) {
    $timestamp = time();
    return url("user/reset/$account->uid/$timestamp/" . user_pass_rehash($account->pass, $timestamp, $account->login), array('absolute' => TRUE));
  }

  /**
   * @Given /^User with name "([^"]*)" and roles "([^"]*)" exists$/
   *
   * @param string $name
   *   login username
   * @param string $roles_string
   *   comma separated list of roles
   */
  public function drupalCreateUser($name, $roles_string) {
    $roles = array();
    foreach (explode(',', $roles_string) as $role) {
      $role = trim($role);
      $role_obj = user_role_load_by_name($role);
      if (empty($role_obj)) {
        throw new Exception(t('Role @role does not exist', array('@role' => $role)));
      }
      $roles[] = $role_obj;
    }

    $rids = array();
    foreach ($roles as $role) {
      $rids[$role->rid] = $role->rid;
    }

    // Create a user assigned to that role.
    $edit = array();
    $edit['name']   = $name;
    $edit['mail']   = $edit['name'] . '@example.com';
    $edit['pass']   = user_password();
    $edit['status'] = 1;

    if (!empty($rids)) {
      $edit['roles'] = $rids;
    }

    $account = user_save(drupal_anonymous_user(), $edit);

    if (empty($account->uid)) {
      return FALSE;
    }

    // Add the raw password so that we can log in as this user.
    $account->pass_raw = $edit['pass'];

    // Reference user above.
    $this->users[$account->uid] = $account;

    return $account;
  }

  /**
   * Generates a random string containing letters and numbers.
   *
   * The string will always start with a letter. The letters may be upper or
   * lower case. This method is better for restricted inputs that do not
   * accept certain characters. For example, when testing input fields that
   * require machine readable values (i.e. without spaces and non-standard
   * characters) this method is best.
   *
   * Do not use this method when testing unvalidated user input. Instead, use
   * DrupalWebTestCase::randomString().
   *
   * @param int $length
   *   Length of random string to generate.
   *
   * @return string
   *   Randomly generated string.
   */
  public function randomName($length = 8) {
    $values = array_merge(range(65, 90), range(97, 122), range(48, 57));
    $max = count($values) - 1;
    $str = chr(mt_rand(97, 122));
    for ($i = 1; $i < $length; $i++) {
      $str .= chr($values[mt_rand(0, $max)]);
    }
    return $str;
  }

}
