<?php
/**
 * @file
 * Various functions
 *
 * This file is largely based upon Drupal OpenID core and provides the
 * the actual authentication functionalities
 */

/**
 * OpenID extensions
 */
define('BEIDIDP_NS_AX', 'http://openid.net/srv/ax/1.0');

module_load_include('inc', 'openid');
module_load_include('inc', 'openid', 'xrds');


/**
 * Hash unique Rijksregister Nummer: storing the RRN in plain text without
 * permission of the Privacy Commission is against Belgian law.
 *
 * @param string $plain
 *   RRN in plain text
 *
 * @return string
 *   Hashed RRN
 */
function beididp_hash($plain) {
  // Change these variables in settings.php
  $prefix = variable_get('beididp_hash_prefix', 'pUJ122deM0');
  $suffix = variable_get('beididp_hash_suffix', 'gwct5ZopY7');
  return sha1($prefix . $plain . $suffix);
}


/**
 * Extract openid namespaces.
 *
 * @param array $response
 *   Response from IDP.
 * @param string $extension_namespace
 *   Namespace of the OpenID extension.
 * @param string $fallback_prefix
 *   Fallback namespace prefix
 *
 * @return array
 *   Array with OpenID extension key/values
 */
function beididp_extract_namespace($response, $extension_namespace, $fallback_prefix = NULL) {
  // Find the namespace prefix.
  $prefix = $fallback_prefix;
  foreach ($response as $key => $value) {
    if ($value == $extension_namespace && preg_match('/^openid\.ns\.([^.]+)$/', $key, $matches)) {
      $prefix = $matches[1];
      break;
    }
  }
  // Now extract the namespace keys from the response.
  $output = array();
  if (!isset($prefix)) {
    return $output;
  }
  foreach ($response as $key => $value) {
    if (preg_match('/^openid\.' . $prefix . '\.(.+)$/', $key, $matches)) {
      $local_key = $matches[1];
      $output[$local_key] = $value;
    }
  }

  return $output;
}

/**
 * Create a username from user's first and lastname (as stored on the eID)
 *
 * @param string $first_name
 *   First name of the citizen
 * @param string $last_name
 *   Last name of the citizen
 *
 * @return string
 *   Concatenated string
 */
function beididp_name_nick($first_name, $last_name) {
  return preg_replace('/\W/', '', $first_name . $last_name);
}

/**
 * Login form _validate hook
 */
function beididp_init_openid(&$form_state) {
  $return_to = $form_state['values']['openid.return_to'];

  if (empty($return_to)) {
    $return_to = url('', array('absolute' => TRUE));
  }

  $idp = variable_get('beididp_idp_url', 'https://www.e-contract.be/eid-idp/endpoints/openid/auth-ident');
  beididp_begin($idp, $return_to, $form_state['values']);
}


/**
 * The initial step of OpenID authentication responsible for the following:
 *  - Perform discovery on the claimed OpenID.
 *  - If possible, create an association with the Provider's endpoint.
 *  - Create the authentication request.
 *  - Perform the appropriate redirect.
 *
 * @param string $claimed_id
 *   The OpenID to authenticate
 * @param string $return_to
 *   The endpoint to return to from the OpenID Provider
 * @param array $form_values
 *   Login form values
 */
function beididp_begin($claimed_id, $return_to = '', $form_values = array()) {
  // Pressflow (and D7+) requires you to explicitly start the session for
  // anonymous visitors.
  // @see http://drupal.org/node/201122
  if (function_exists('drupal_session_start')) {
    drupal_session_start();
  }
  $claimed_id = _openid_normalize($claimed_id);

  $services = beididp_discovery($claimed_id);
  if (count($services) == 0) {
    form_set_error('openid_identifier', t('Sorry, that is not a valid OpenID. Please ensure you have spelled your ID correctly.'));
    return;
  }

  // Store discovered info in the users' session so we don't have to rediscover.
  $_SESSION['openid']['service'] = $services[0];
  $_SESSION['openid']['claimed_id'] = $claimed_id;

  // Store login form values so we can pass them to user_exteral_login later.
  $_SESSION['openid']['user_login_values'] = $form_values;

  $op_endpoint = $services[0]['uri'];
  $assoc_handle = beididp_association($op_endpoint);

  // First check for LocalID. If not found, check for Delegate.
  // Fall back to $claimed_id if neither is found.
  if (!empty($services[0]['localid'])) {
    $identity = $services[0]['localid'];
  }
  elseif (!empty($services[0]['delegate'])) {
    $identity = $services[0]['delegate'];
  }
  else {
    $identity = $claimed_id;
  }

  if (isset($services[0]['types']) && is_array($services[0]['types']) && in_array(OPENID_NS_2_0 . '/server', $services[0]['types'])) {
    $claimed_id = $identity = 'http://specs.openid.net/auth/2.0/identifier_select';
  }
  $authn_request = beididp_authentication_request($claimed_id, $identity, $return_to, $assoc_handle, $services[0]['version']);

  openid_redirect($op_endpoint, $authn_request);
}


/**
 * Completes OpenID authentication by validating returned data from the OpenID
 * Provider.
 *
 * @param array $response
 *   Array of returned values from the OpenID Provider.
 *
 * @return array
 *   Response values for further processing with
 *   $response['status'] set to one of 'success', 'failed' or 'cancel'.
 */
function beididp_complete($response = array()) {
  if (count($response) == 0) {
    $response = _openid_response();
  }

  // Default to failed response.
  $response['status'] = 'failed';
  if (isset($_SESSION['openid']['service']['uri']) && isset($_SESSION['openid']['claimed_id'])) {
    $service = $_SESSION['openid']['service'];
    $claimed_id = $_SESSION['openid']['claimed_id'];
    unset($_SESSION['openid']['service']);
    unset($_SESSION['openid']['claimed_id']);

    if (isset($response['openid.mode'])) {
      if ($response['openid.mode'] == 'cancel') {
        watchdog('beididp', 'Cancel response received (claimed ID = @id)',
                 array('@id' => $claimed_id), WATCHDOG_WARNING);
        $response['status'] = 'cancel';
      }
      else {
        if (beididp_verify_assertion($service['uri'], $response)) {
          // If the returned claimed_id is different from session claimed_id,
          // then we need to do discovery and make sure op_endpoint matches.
          if ($service['version'] == 2) {
            $response['openid.claimed_id'] = _openid_normalize($response['openid.claimed_id']);
            // OpenID Authentication, section 11.2:
            // If the returned Claimed Identifier is different from the one sent
            // to the OpenID Provider, we need to do discovery on the returned
            // identifier to make sure that the provider is authorized to
            // respond on behalf of this.
            if ($response['openid.claimed_id'] != $claimed_id) {
              $services = beididp_discovery($response['openid.claimed_id']);
              $uris = array();
              foreach ($services as $discovered_service) {
                if (in_array('http://specs.openid.net/auth/2.0/server', $discovered_service['types']) || in_array('http://specs.openid.net/auth/2.0/signon', $discovered_service['types'])) {
                  $uris[] = $discovered_service['uri'];
                }
              }
              if (!in_array($service['uri'], $uris)) {
                watchdog('beididp',
                  'Received ID @recv does not match claimed ID @id',
                  array('@id' => $claimed_id, '@recv' => $response['openid.claimed_id']),
                  WATCHDOG_ERROR);

                return $response;
              }
            }
          }
          else {
            $response['openid.claimed_id'] = $claimed_id;
          }
          $response['status'] = 'success';
        }
      }
    }
    else {
      watchdog('beididp', 'Response mode not set');
    }
  }

  // Optionally hash the claimed_id (since it contains Rijksregister Nummer).
  if (variable_get('beididp_beididp_hash_claimed_id', TRUE) && isset($response['openid.claimed_id'])) {
    $pattern = "/(.+\?)(\d{11})$/";
    if (preg_match($pattern, $response['openid.claimed_id'], $matches) > 0 && is_array($matches) && count($matches) == 3) {
      $hash = $matches[1] . beididp_hash($matches[2]);
    }
    else {
      $hash = beididp_hash($response['openid.claimed_id']);
    }
    $response['openid.claimed_id'] = $hash;
  }
  return $response;
}

/**
 * Perform discovery on a claimed ID to determine the OpenID provider endpoint.
 *
 * @param string $claimed_id
 *   The OpenID URL to perform discovery on.
 *
 * @return array
 *   Array of services discovered (including endpoint URI etc).
 */
function beididp_discovery($claimed_id) {
  $services = array();

  $xrds_url = $claimed_id;
  if (_openid_is_xri($claimed_id)) {
    $xrds_url = 'http://xri.net/' . $claimed_id;
  }
  $url = @parse_url($xrds_url);
  if ($url['scheme'] === 'https') {
    // For regular URLs, try Yadis resolution first, then HTML-based discovery.
    $headers = array('Accept' => 'application/xrds+xml');
    $result = drupal_http_request($xrds_url, $headers);

    if (!isset($result->error)) {
      if (isset($result->headers['Content-Type']) && preg_match("/application\/xrds\+xml/", $result->headers['Content-Type'])) {
        // Parse XML document to find URL.
        $services = xrds_parse($result->data);
      }
      else {
        $xrds_url = NULL;
        if (isset($result->headers['X-XRDS-Location'])) {
          $xrds_url = $result->headers['X-XRDS-Location'];
        }
        else {
          // Look for meta http-equiv link in HTML head.
          $xrds_url = _openid_meta_httpequiv('X-XRDS-Location', $result->data);
        }
        if (!empty($xrds_url)) {
          $headers = array('Accept' => 'application/xrds+xml');
          $xrds_result = drupal_http_request($xrds_url, $headers);
          if (!isset($xrds_result->error)) {
            $services = xrds_parse($xrds_result->data);
          }
        }
      }
    }
    else {
      drupal_set_message(t('XRDS error: @error', array('@error' => $result->error)), 'error');
    }
  }
  else {
    drupal_set_message(t('URL scheme is not https'), 'error');
  }
  return $services;
}

/**
 * Attempt to create a shared secret with the OpenID Provider.
 *
 * @param string $op_endpoint
 *   URL of the OpenID Provider endpoint.
 *
 * @return string
 *   The association handle.
 */
function beididp_association($op_endpoint) {
  // Remove Old Associations:
  db_query("DELETE FROM {beididp_association} WHERE created + expires_in < %d", time());

  // Check to see if we have an association for this IdP already.
  $assoc_handle = db_result(db_query("SELECT assoc_handle FROM {beididp_association} WHERE idp_endpoint_uri = '%s'", $op_endpoint));
  if (empty($assoc_handle)) {
    $mod = OPENID_DH_DEFAULT_MOD;
    $gen = OPENID_DH_DEFAULT_GEN;
    $r = _openid_dh_rand($mod);
    $private = bcadd($r, 1);
    $public = bcpowmod($gen, $private, $mod);

    // If there is no existing association, then request one.
    $assoc_request = beididp_association_request($public);
    $assoc_message = _openid_encode_message(_openid_create_message($assoc_request));
    $assoc_headers = array('Content-Type' => 'application/x-www-form-urlencoded; charset=utf-8');
    $assoc_result = drupal_http_request($op_endpoint, $assoc_headers, 'POST', $assoc_message);
    if (isset($assoc_result->error)) {
      return FALSE;
    }

    $assoc_response = _openid_parse_message($assoc_result->data);
    if (isset($assoc_response['mode']) && $assoc_response['mode'] == 'error') {
      return FALSE;
    }

    if ($assoc_response['session_type'] == 'DH-SHA1') {
      $spub = _openid_dh_base64_to_long($assoc_response['dh_server_public']);
      $enc_mac_key = base64_decode($assoc_response['enc_mac_key']);
      $shared = bcpowmod($spub, $private, $mod);
      $assoc_response['mac_key'] = base64_encode(_openid_dh_xorsecret($shared, $enc_mac_key));
    }
    $record = array(
      'idp_endpoint_uri' => $op_endpoint,
      'session_type' => $assoc_response['session_type'],
      'assoc_handle' => $assoc_response['assoc_handle'],
      'assoc_type' => $assoc_response['assoc_type'],
      'expires_in' => $assoc_response['expires_in'],
      'mac_key' => $assoc_response['mac_key'],
      'created' => time(),
    );
    drupal_write_record('beididp_association', $record);
    $assoc_handle = $assoc_response['assoc_handle'];
  }

  return $assoc_handle;
}

/**
 * Authenticate a user or attempt registration.
 *
 * @param array $response
 *   Response values from the OpenID Provider.
 */
function beididp_authentication($response) {
  $identity = $response['openid.claimed_id'];

  $account = user_external_load($identity);
  if (isset($account->uid)) {
    $skip_verify = variable_get('beididp_no_mail_verify', TRUE);
    $verify = variable_get('user_email_verification', TRUE);
    if (!$verify || $account->login || $skip_verify) {
      user_external_login($account, $_SESSION['openid']['user_login_values']);
    }
    else {
      drupal_set_message(t('You must validate your email address for this account before logging in via OpenID'));
    }
  }
  elseif (variable_get('user_register', 1)) {
    $_SESSION['openid']['values']['response'] = $response;
    unset($_REQUEST['destination']);
    drupal_goto('user/register', "beididp/authenticate");
  }
  else {
    drupal_set_message(t('Only site administrators can create new user accounts.'), 'error');
  }
  drupal_goto();
}

/**
 * Association request.
 *
 * @param long $public
 *   Public key.
 *
 * @return array
 *   Array with request parameters.
 */
function beididp_association_request($public) {
  $request = array(
    'openid.ns' => OPENID_NS_2_0,
    'openid.mode' => 'associate',
    'openid.session_type' => 'DH-SHA1',
    'openid.assoc_type' => 'HMAC-SHA1',
  );

  if ($request['openid.session_type'] == 'DH-SHA1' || $request['openid.session_type'] == 'DH-SHA256') {
    $cpub = _openid_dh_long_to_base64($public);
    $request['openid.dh_consumer_public'] = $cpub;
  }

  return $request;
}

/**
 * Authentication request.
 *
 * @param string $claimed_id
 *   Claimed ID.
 * @param string $identity
 *   Identity.
 * @param string $return_to
 *   URL to return to after authentication on the IDP.
 * @param string $assoc_handle
 *   Association handle.
 */
function beididp_authentication_request($claimed_id, $identity, $return_to = '', $assoc_handle = '') {
  global $language;
  $lang_name = $language->language;

  $request = array(
    'openid.ns' => OPENID_NS_2_0,
    'openid.ns.ui' => 'http://specs.openid.net/extensions/ui/1.0',
    'openid.ns.ax' => BEIDIDP_NS_AX,
    'openid.mode' => 'checkid_setup',
    'openid.identity' => $identity,
    'openid.claimed_id' => $claimed_id,
    'openid.assoc_handle' => $assoc_handle,
    'openid.return_to' => $return_to,
    'openid.realm' => url('', array('absolute' => TRUE)),
    'openid.ui.lang' => $lang_name . '-BE',
    'openid.ax.mode' => 'fetch_request',
    'openid.ax.type.fname' => 'http://axschema.org/namePerson/first',
    'openid.ax.type.lname' => 'http://axschema.org/namePerson/last',
    'openid.ax.type.mname' => 'http://axschema.org/namePerson/middle',
    'openid.ax.required' => 'fname,lname',
    'openid.ax.if_available' => 'mname',
  );

  $request = array_merge($request, module_invoke_all('beididp_auth_request', $request));
  return $request;
}

/**
 * Attempt to verify the response received from the OpenID Provider.
 *
 * @param string $op_endpoint
 *   The OpenID Provider URL.
 * @param array $response
 *   Array of response values from the provider.
 *
 * @return bool
 *   Boolean
 */
function beididp_verify_assertion($op_endpoint, $response) {
  $valid = FALSE;
  $association = FALSE;

  if (!beididp_verify_assertion_nonce($op_endpoint, $response)) {
    return FALSE;
  }

  // If the OP returned a openid.invalidate_handle, we have to proceed with
  // direct verification: ignore the openid.assoc_handle, even if present.
  // See OpenID spec 11.4.1
  if (!empty($response['openid.assoc_handle']) && empty($response['openid.invalidate_handle'])) {
     $association = db_fetch_object(db_query("SELECT * FROM {beididp_association} WHERE assoc_handle = '%s'", $response['openid.assoc_handle']));
  }

  if ($association && isset($association->session_type)) {
    $keys_to_sign = explode(',', $response['openid.signed']);
    $self_sig = _openid_signature($association, $response, $keys_to_sign);
    if ($self_sig == $response['openid.sig']) {
      $valid = TRUE;
    }
    else {
      watchdog('beididp', 'Invalid association signature (handle = @handle)',
               array('@handle' => $response['openid.assoc_handle']),
               WATCHDOG_ERROR);
      $valid = FALSE;
    }
  }
  else {
    // See OpenID spec 11.4.2.1
    // The verification requests contain all the fields from the response,
    // except openid.mode.
    $request = $response;
    $request['openid.mode'] = 'check_authentication';
    $message = _openid_create_message($request);
    $headers = array('Content-Type' => 'application/x-www-form-urlencoded; charset=utf-8');
    $result = drupal_http_request($op_endpoint, $headers, 'POST', _openid_encode_message($message));
    if (!isset($result->error)) {
      $response = _openid_parse_message($result->data);
      if (drupal_strtolower(trim($response['is_valid'])) == 'true') {
        $valid = TRUE;
        if (!empty($response['invalidate_handle'])) {
          // This association handle has expired on the OP side, remove it
          // from the database to avoid reusing it again on a subsequent
          // authentication request.
          // See OpenID spec 11.4.2.2
          db_query("DELETE FROM {beididp_association} WHERE assoc_handle = '%s'", $response['invalidate_handle']);
        }
      }
      else {
        watchdog('beididp', 'Association error @code @err (claimed id @id)',
               array('@err' => $result->error, '@code' => $result->code, '@id' => $response['openid.claimed_id']),
               WATCHDOG_ERROR);
        $valid = FALSE;
      }
    }
  }

  // Additional checks.
  if ($valid) {
    $results = module_invoke_all('beididp_verify_assert', $response);
    if ((!empty($results)) && in_array(FALSE, $results)) {
      watchdog('beididp', 'Additional verification failed (claimed id @id)',
               array('@id' => $response['openid.claimed_id']), WATCHDOG_ERROR);
      $valid = FALSE;
    }
  }

  return $valid;
}

/**
 * Verify nonce.
 *
 * @param array $service
 *   Service
 * @param array $response
 *   OpenID response
 *
 * @return bool
 *   Boolean
 */
function beididp_verify_assertion_nonce($service, $response) {
  if (preg_match('/^(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2})Z/', $response['openid.response_nonce'], $matches)) {
    list(, $year, $month, $day, $hour, $minutes, $seconds) = $matches;
    $nonce_timestamp = gmmktime($hour, $minutes, $seconds, $month, $day, $year);
  }
  else {
    watchdog('beididp', 'Nonce from @endpoint rejected because it is not correctly formatted, nonce: @nonce.', array('@endpoint' => $service, '@nonce' => $response['openid.response_nonce']), WATCHDOG_WARNING);
    return FALSE;
  }

  // A nonce with a timestamp too far in the past or future will already have
  // been removed and cannot be checked for single use anymore.
  $time = time();
  $expiry = 900;
  if ($nonce_timestamp <= $time - $expiry || $nonce_timestamp >= $time + $expiry) {
    watchdog('beididp', 'Nonce received from @endpoint is out of range (time difference: @intervals). Check possible clock skew.', array('@endpoint' => $service, '@interval' => $time - $nonce_timestamp), WATCHDOG_WARNING);
    return FALSE;
  }

  // Record that this nonce was used.
  $record = array(
    'idp_endpoint_uri' => $service,
    'nonce' => $response['openid.response_nonce'],
    'expires' => $nonce_timestamp + $expiry,
  );
  drupal_write_record('beididp_nonce', $record);

  // Count the number of times this nonce was used.
  $count_used = db_result(db_query("SELECT COUNT(*) FROM {beididp_nonce} WHERE nonce = '%s' AND idp_endpoint_uri = '%s'", $response['openid.response_nonce'], $service));

  if ($count_used == 1) {
    return TRUE;
  }
  else {
    watchdog('beididp', 'Nonce replay attempt blocked from @ip, service: @service, nonce: @nonce.', array('@ip' => ip_address(), '@service' => $service, '@nonce' => $response['openid.response_nonce']), WATCHDOG_CRITICAL);
    return FALSE;
  }
}
