<?php
/**
 * @file
 * Various functions
 *
 * This file is largely based upon Drupal OpenID core and provides the
 * the actual authentication functionalities
 */

/**
 * OpenID extensions
 */
define('BEIDIDP_NS_AX', 'http://openid.net/srv/ax/1.0');

module_load_include('module', 'openid');
module_load_include('inc', 'openid');
module_load_include('inc', 'openid', 'xrds');


/**
 * Hash unique Rijksregister Nummer.
 *
 * Storing the RRN in plain text without permission of the Privacy Commission
 * is against Belgian law.
 *
 * @param string $plain
 *   RRN in plain text
 *
 * @return string
 *   Hashed RRN
 */
function beididp_hash($plain) {
  // Change these variables in settings.php
  $prefix = variable_get('beididp_hash_prefix', 'pUJ122deM0');
  $suffix = variable_get('beididp_hash_suffix', 'gwct5ZopY7');
  return sha1($prefix . $plain . $suffix);
}


/**
 * Hash the citizen's unique Rijksregister number.
 *
 * @param string $claimed_id
 *   URL containing the (not hashed) RRN
 *
 * @return string
 *   URL containing the hashed RRN
 */
function beididp_hash_rrn($claimed_id) {
  $pattern = "/(.+\?)(\d{11})$/";

  if (preg_match($pattern, $claimed_id, $matches) > 0 && is_array($matches) && count($matches) == 3) {
    $hash = $matches[1] . beididp_hash($matches[2]);
  }
  else {
    $hash = beididp_hash($claimed_id);
  }

  return $hash;
}

/**
 * Create a username.
 *
 * Use the citizen's first and last name (as stored on the eID).
 * If this name is already taken, then add a numeric suffix.
 *
 * @param string $first_name
 *   First name of the citizen
 * @param string $last_name
 *   Last name of the citizen
 *
 * @return string
 *   Concatenated string
 */
function beididp_name_nick($first_name, $last_name) {
  $name = preg_replace('/\W/', '', $first_name . $last_name);
  $nick = $name;
  $suffix = 1;

  while (beididp_user_exists($nick)) {
    $nick = $name . $suffix++;
  }
  return $nick;
}


/**
 * Check if a certain user name already exists.
 *
 * @param string $name
 *   User name to be checked
 *
 * @return bool
 *   TRUE if a user name is already in the database
 */
function beididp_user_exists($name) {
  return db_query('SELECT 1 FROM {users} WHERE name = :name', array(':name' => $name))->fetchColumn();
}


/**
 * The initial step of OpenID authentication.
 *
 * This is responsible for the following:
 *  - Perform discovery on the claimed OpenID.
 *  - If possible, create an association with the Provider's endpoint.
 *  - Create the authentication request.
 *  - Perform the appropriate redirect.
 *
 * @param string $return_to
 *   The endpoint to return to from the OpenID Provider
 * @param array $form_values
 *   Login form values
 */
function beididp_begin($return_to = '', $form_values = array()) {
  drupal_session_start();

  $idp = variable_get('beididp_idp_url', 'https://www.e-contract.be/eid-idp/endpoints/openid/auth-ident');
  $idp = openid_normalize($idp);

  $services = beididp_discovery($idp);
  if (empty($services)) {
    form_set_error('openid_identifier', t('Not a valid eID IDP service.'));
    watchdog('beididp', 'No services discovered on IDP @idp)',
      array('@idp' => $idp), WATCHDOG_ERROR);
    return;
  }

  if (isset($services[0]['types']) && is_array($services[0]['types']) && in_array(OPENID_NS_2_0 . '/server', $services[0]['types'])) {
    // Store discovered info in the users' session.
    $_SESSION['beididp']['service']['uri'] = $services[0]['uri'];
    $_SESSION['beididp']['service']['version'] = 2;
    $_SESSION['beididp']['claimed_id'] = $idp;

    // Store login form values so we can pass them to user_authenticate
    $_SESSION['beididp']['user_login_values'] = $form_values;

    $op_endpoint = $services[0]['uri'];
    $assoc_handle = beididp_association($op_endpoint);

    $claimed_id = $identity = OPENID_NS_2_0 . '/identifier_select';
    $authn_request = beididp_authentication_request($claimed_id, $identity, $return_to, $assoc_handle);

    openid_redirect($op_endpoint, $authn_request);
  }
  else {
    form_set_error('openid_identifier', t('Not a valid eID service.'));
    watchdog('beididp', 'Error in service type from IDP @idp)',
      array('@idp' => $idp), WATCHDOG_ERROR);
  }
}


/**
 * Completes OpenID authentication.
 *
 * Validates returned data from the Belgian eID IDP / OpenID Provider.
 *
 * @return array
 *   Response values for further processing with
 *   $response['status'] set to one of 'success', 'failed' or 'cancel'.
 */
function beididp_complete() {
  $response = _openid_response();

  // Default to failed response.
  $response['status'] = 'failed';

  if (isset($_SESSION['beididp']['service']['uri']) && isset($_SESSION['beididp']['claimed_id'])) {
    $service = $_SESSION['beididp']['service'];
    $claimed_id = $_SESSION['beididp']['claimed_id'];
    unset($_SESSION['beididp']['service']);
    unset($_SESSION['beididp']['claimed_id']);
  }
  else {
    watchdog('beididp', 'Session variables beididp not set', array(), WATCHDOG_WARNING);
    return $response;
  }

  if (!isset($response['openid.mode'])) {
    watchdog('beididp', 'Response mode not not set', array(), WATCHDOG_ERROR);
    return $response;
  }

  if ($response['openid.mode'] == 'cancel') {
    watchdog('beididp', 'Cancel response received (claimed ID = @id)',
      array('@id' => $claimed_id), WATCHDOG_WARNING);
    $response['status'] = 'cancel';

    return $response;
  }

  if (!beididp_verify_assertion($service, $response)) {
    watchdog('beididp', 'Assertion could not be verified (claimed ID = @id)',
      array('@id' => $claimed_id), WATCHDOG_ERROR);
    return $response;
  }

  $response['openid.claimed_id'] = openid_normalize($response['openid.claimed_id']);

  // OpenID Authentication, section 11.2:
  // If the returned Claimed Identifier is different from the one sent
  // to the OpenID Provider, we need to do discovery on the returned
  // identifier to make sure that the provider is authorized to
  // respond on behalf of this.
  if ($response['openid.claimed_id'] != $claimed_id) {
    $services = beididp_discovery($response['openid.claimed_id']);
    $uris = array();

    foreach ($services as $discovered) {
      if (in_array(OPENID_NS_2_0 . '/server', $discovered['types']) || in_array(OPENID_NS_2_0 . '/signon', $discovered['types'])) {
        $uris[] = $discovered['uri'];
      }
    }

    if (!in_array($service['uri'], $uris)) {
      watchdog('beididp', 'Received ID @recv does not match claimed ID @id',
        array('@id' => $claimed_id, '@recv' => $response['openid.claimed_id']),
        WATCHDOG_ERROR);

      return $response;
    }
  }
  else {
    $response['openid.claimed_id'] = $claimed_id;
  }

  $response['status'] = 'success';

  // Optionally hash the claimed_id (since it contains Rijksregister Nummer).
  if (variable_get('beididp_beididp_hash_claimed_id', TRUE) && isset($response['openid.claimed_id'])) {
    $response['openid.claimed_id'] = beididp_hash_rrn($response['openid.claimed_id']);
  }
  return $response;
}


/**
 * Perform discovery on a claimed ID to determine the OpenID provider endpoint.
 *
 * @param string $claimed_id
 *   The OpenID URL to perform discovery on.
 *
 * @return array
 *   Array of services discovered (including endpoint URI etc).
 */
function beididp_discovery($claimed_id) {
  $services = array();
  $url = @parse_url($claimed_id);

  if ($url['scheme'] === 'https') {
    $xrds = _openid_xrds_discovery($claimed_id);
    if (!isset($xrds) || empty($xrds)) {
      watchdog('beididp', 'No services discovered for @id',
        array('@id' => $claimed_id), WATCHDOG_ERROR);
    }
    $services = $xrds['services'];
  }
  else {
    drupal_set_message(t('Discovery scheme is not https'), 'error');
    watchdog('beididp', 'Discovery scheme is not in https for @id',
        array('@id' => $claimed_id), WATCHDOG_ERROR);
  }
  return $services;
}

/**
 * Attempt to create a shared secret with the OpenID Provider.
 *
 * @param string $op_endpoint
 *   URL of the OpenID Provider endpoint.
 *
 * @return string
 *   The association handle.
 */
function beididp_association($op_endpoint) {
  // Remove old associations.
  db_delete('beididp_association')
    ->where('created + expires_in < :time', array(':time' => REQUEST_TIME))
    ->execute();

  // Check to see if we have an association for this IdP already.
  $assoc_handle = db_query("SELECT assoc_handle FROM {beididp_association} WHERE idp_endpoint_uri = :endpoint", array(':endpoint' => $op_endpoint))->fetchColumn();
  if (empty($assoc_handle)) {
    $mod = OPENID_DH_DEFAULT_MOD;
    $gen = OPENID_DH_DEFAULT_GEN;
    $r = _openid_dh_rand($mod);
    $private = _openid_math_add($r, 1);
    $public = _openid_math_powmod($gen, $private, $mod);

    // If there is no existing association, then request one.
    $assoc_request = openid_association_request($public);
    $assoc_message = _openid_encode_message(_openid_create_message($assoc_request));
    $assoc_options = array(
      'headers' => array('Content-Type' => 'application/x-www-form-urlencoded; charset=utf-8'),
      'method' => 'POST',
      'data' => $assoc_message,
    );
    $assoc_result = drupal_http_request($op_endpoint, $assoc_options);
    if (isset($assoc_result->error)) {
      return FALSE;
    }

    $assoc_response = _openid_parse_message($assoc_result->data);
    if (isset($assoc_response['mode']) && $assoc_response['mode'] == 'error') {
      return FALSE;
    }

    if ($assoc_response['session_type'] == 'DH-SHA1') {
      $spub = _openid_dh_base64_to_long($assoc_response['dh_server_public']);
      $enc_mac_key = base64_decode($assoc_response['enc_mac_key']);
      $shared = bcpowmod($spub, $private, $mod);
      $assoc_response['mac_key'] = base64_encode(_openid_dh_xorsecret($shared, $enc_mac_key));
    }
    db_insert('beididp_association')
      ->fields(array(
        'idp_endpoint_uri' => $op_endpoint,
        'session_type' => $assoc_response['session_type'],
        'assoc_handle' => $assoc_response['assoc_handle'],
        'assoc_type' => $assoc_response['assoc_type'],
        'expires_in' => $assoc_response['expires_in'],
        'mac_key' => $assoc_response['mac_key'],
        'created' => REQUEST_TIME,
    ))
    ->execute();
    $assoc_handle = $assoc_response['assoc_handle'];
  }

  return $assoc_handle;
}

/**
 * Authenticate a user or attempt registration.
 *
 * @param array $response
 *   Response values from the OpenID Provider.
 */
function beididp_authentication($response) {
  $identity = $response['openid.claimed_id'];

  $account = user_external_load($identity);
  if (isset($account->uid)) {
    $verify = variable_get('user_email_verification', TRUE);
    if ($verify) {
      $verify = !variable_get('beididp_no_mail_verify', TRUE);
    }

    if (!$verify || $account->login) {
      $state['values']['name'] = $account->name;
      user_login_name_validate(array(), $state);
      if (!form_get_errors()) {
        $form_state['uid'] = $account->uid;
        user_login_submit(array(), $form_state);
      }
    }
    else {
      drupal_set_message(t('You must validate your email address for this account before logging in via eID'));
    }
  }
  elseif (variable_get('user_register', USER_REGISTER_VISITORS)) {
    $_SESSION['beididp']['values']['response'] = $response;
    $destination = drupal_get_destination();
    unset($_GET['destination']);
    drupal_goto('user/register', $destination);
  }
  else {
    drupal_set_message(t('Only site administrators can create new user accounts.'), 'error');
  }
  drupal_goto();
}

/**
 * Authentication request.
 *
 * @param string $claimed_id
 *   Claimed ID.
 * @param string $identity
 *   Identity.
 * @param string $return_to
 *   URL to return to after authentication on the IDP.
 * @param string $assoc_handle
 *   Association handle.
 */
function beididp_authentication_request($claimed_id, $identity, $return_to = '', $assoc_handle = '') {
  global $language;
  $lang_name = $language->language;

  $request = array(
    'openid.ns' => OPENID_NS_2_0,
    'openid.ns.ui' => 'http://specs.openid.net/extensions/ui/1.0',
    'openid.ns.ax' => BEIDIDP_NS_AX,
    'openid.mode' => 'checkid_setup',
    'openid.identity' => $identity,
    'openid.claimed_id' => $claimed_id,
    'openid.assoc_handle' => $assoc_handle,
    'openid.return_to' => $return_to,
    'openid.realm' => url('', array('absolute' => TRUE)),
    'openid.ui.lang' => $lang_name . '-BE',
    'openid.ax.mode' => 'fetch_request',
    'openid.ax.type.fname' => 'http://axschema.org/namePerson/first',
    'openid.ax.type.lname' => 'http://axschema.org/namePerson/last',
    'openid.ax.type.mname' => 'http://axschema.org/namePerson/middle',
    'openid.ax.required' => 'fname,lname',
    'openid.ax.if_available' => 'mname',
  );

  $request = array_merge($request, module_invoke_all('beididp_auth_request', $request));
  return $request;
}

/**
 * Attempt to verify the response received from the OpenID Provider.
 *
 * @param string $service
 *   The OpenID Provider URL.
 * @param array $response
 *   Array of response values from the provider.
 *
 * @return bool
 *   Boolean
 */
function beididp_verify_assertion($service, $response) {
  $valid = FALSE;
  $association = FALSE;
  $op_endpoint = $service['uri'];

  if (!beididp_verify_assertion_nonce($op_endpoint, $response)) {
    return FALSE;
  }

  if (!openid_verify_assertion_return_url($op_endpoint, $response)) {
    watchdog('beididp', 'Return_to URL @url assertion failed',
      array('@url' => $response['openid.return_to']), WATCHDOG_ERROR);
    return FALSE;
  }

  // If the OP returned a openid.invalidate_handle, we have to proceed with
  // direct verification: ignore the openid.assoc_handle, even if present.
  // See OpenID spec 11.4.1
  if (!empty($response['openid.assoc_handle']) && empty($response['openid.invalidate_handle'])) {
    $association = db_query("SELECT * FROM {beididp_association} WHERE assoc_handle = :handle", array(':handle' => $response['openid.assoc_handle']))->fetchObject();
  }

  if ($association && isset($association->session_type)) {
    $valid = openid_verify_assertion_signature($service, $association, $response);
    if (!$valid) {
      watchdog('beididp', 'Invalid association signature (handle = @handle)',
        array('@handle' => $response['openid.assoc_handle']),
        WATCHDOG_ERROR);
    }
  }
  else {
    // See OpenID spec 11.4.2.1
    // The verification requests contain all the fields from the response,
    // except openid.mode.
    $request = $response;
    $request['openid.mode'] = 'check_authentication';
    $message = _openid_create_message($request);
    $options = array(
      'headers' => array('Content-Type' => 'application/x-www-form-urlencoded; charset=utf-8'),
      'method' => 'POST',
      'data' => _openid_encode_message($message),
    );
    $result = drupal_http_request($op_endpoint, $options);
    if (!isset($result->error)) {
      $response = _openid_parse_message($result->data);
      if (drupal_strtolower(trim($response['is_valid'])) == 'true') {
        $valid = TRUE;
        if (!empty($response['invalidate_handle'])) {
          // This association handle has expired on the OP side, remove it
          // from the database to avoid reusing it again on a subsequent
          // authentication request.
          // See OpenID spec 11.4.2.2
          db_delete('beididp_association')
            ->condition('assoc_handle', $response['invalidate_handle'])
            ->execute();
        }
      }
      else {
        watchdog('beididp', 'Association error @code @err (claimed id @id)',
          array(
            '@err' => $result->error,
            '@code' => $result->code,
            '@id' => $response['openid.claimed_id'],
          ),
          WATCHDOG_ERROR);
        $valid = FALSE;
      }
    }
  }

  // Additional checks.
  if ($valid) {
    $results = module_invoke_all('beididp_verify_assert', $response);
    if ((!empty($results)) && in_array(FALSE, $results)) {
      watchdog('beididp', 'Additional verification failed (claimed id @id)',
        array('@id' => $response['openid.claimed_id']), WATCHDOG_ERROR);
      $valid = FALSE;
    }
  }

  return $valid;
}

/**
 * Verify nonce.
 *
 * @param array $service
 *   Service
 * @param array $response
 *   OpenID response
 *
 * @return bool
 *   Boolean
 */
function beididp_verify_assertion_nonce($service, $response) {
  if (preg_match('/^(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2})Z/', $response['openid.response_nonce'], $matches)) {
    list(, $year, $month, $day, $hour, $minutes, $seconds) = $matches;
    $nonce_timestamp = gmmktime($hour, $minutes, $seconds, $month, $day, $year);
  }
  else {
    watchdog('beididp', 'Nonce from @endpoint rejected because it is not correctly formatted, nonce: @nonce.', array('@endpoint' => $service, '@nonce' => $response['openid.response_nonce']), WATCHDOG_ERROR);
    return FALSE;
  }

  // A nonce with a timestamp too far in the past or future will already have
  // been removed and cannot be checked for single use anymore.
  $time = time();
  $expiry = 900;
  if ($nonce_timestamp <= $time - $expiry || $nonce_timestamp >= $time + $expiry) {
    watchdog('beididp', 'Nonce received from @endpoint is out of range (time difference: @intervals). Check possible clock skew.', array('@endpoint' => $service, '@interval' => $time - $nonce_timestamp), WATCHDOG_ERROR);
    return FALSE;
  }

  // Record that this nonce was used.
  $record = array(
    'idp_endpoint_uri' => $service,
    'nonce' => $response['openid.response_nonce'],
    'expires' => $nonce_timestamp + $expiry,
  );
  drupal_write_record('beididp_nonce', $record);

  // Count the number of times this nonce was used.
  $count_used = db_query("SELECT COUNT(*) FROM {beididp_nonce} WHERE nonce = :nonce AND idp_endpoint_uri = :uri", array(':nonce' => $response['openid.response_nonce'], ':uri' => $service))->fetchColumn();

  if ($count_used == 1) {
    return TRUE;
  }
  else {
    watchdog('beididp', 'Nonce replay attempt blocked from @ip, service: @service, nonce: @nonce.',
      array(
        '@ip' => ip_address(),
        '@service' => $service,
        '@nonce' => $response['openid.response_nonce'],
      ),
      WATCHDOG_CRITICAL);
    return FALSE;
  }
}

/**
 * Returns eID identity for a user.
 *
 * @param int $uid
 *   User ID
 *
 * @return DatabaseStatementInterface
 *   A prepared statement object
 */
function beididp_get_eid($uid) {
  return db_query("SELECT * FROM {authmap} WHERE uid = :uid AND module = 'beididp'", array(':uid' => $uid));
}
