/**
 * @file
 * Functionality for Better Social Share popup and sharing links.
 */

(function ($, Drupal, drupalSettings) {
  let isFunctionExecuted = 0;

  // --- Utility Functions defined at the top of the function scope ---

  /**
   * Resets the search input and triggers a keyup event for filtering.
   */
  function resetSearch() {
    setTimeout(function () {
      const searchInput = document.querySelector('input.social-share-search');
      if (searchInput) {
        searchInput.value = '';
        $(searchInput).trigger('keyup');
      }
    }, 1000);
  }

  /**
   * Closes the main social share popup with an animation.
   */
  function hidePopup() {
    const $popupBlock = $('#better-social-share-popup-reg');
    $popupBlock.addClass('is-hiding');

    // Complete closure after animation (600ms matching the CSS transition)
    setTimeout(function () {
      $('.better-social-share-popup').removeClass('active is-visible');
      $popupBlock.removeClass('is-hiding');
      // Cleanup of any previous inline styles (using native JS to avoid linter error)
      $popupBlock.get(0).style.opacity = '';
      $popupBlock
        .find('.better-social-share-popup-content')
        .get(0).style.marginTop = '';
    }, 600);
    resetSearch();
  }

  // --- Drupal Behavior ---

  Drupal.behaviors.betterSocialShare = {
    attach(context, settings) {
      const $context = $(context);

      // ----------------------------------------------------------------------
      // 1. Popup Opening/Closing Logic (Using functional once)
      // ----------------------------------------------------------------------

      // Open Popup on .popup-trigger click
      once(
        'better-social-share-popup-trigger',
        '.popup-trigger',
        context,
      ).forEach(function (triggerElement) {
        const $triggerElement = $(triggerElement);

        $triggerElement.on('click', function (e) {
          e.preventDefault();
          const $popupBlock = $('#better-social-share-popup-reg');
          const $parentAttr = $(this).closest('.social-share-btns');
          const entityUrl = $parentAttr.data('entity_url');
          const entityTitle = $parentAttr.data('entity_title');

          $('.better-social-share-popup-container')
            .data('entity_url', entityUrl)
            .data('entity_title', entityTitle);

          $('.better-social-share-popup').addClass('is-visible');
          $popupBlock.addClass('active');

          // Bind the inner close button click on the fade-out element.
          // Since the popup is static, we can attach this to the document/body once.
          once('fade-out-close', '.fade-out', document).forEach(
            function (closeElement) {
              $(closeElement).on('click', function () {
                hidePopup();
              });
            },
          );
        });
      });

      // Close Popup (delegated click on overlay or close button)
      // Attach to document once as these elements are likely fixed/static.
      once('social-share-close-doc', 'body', document).forEach(
        function (bodyElement) {
          $(bodyElement).on(
            'click',
            '.better-social-share-popup-close, #better-social-share-popup-reg',
            function (event) {
              if (
                event.target.matches('#better-social-share-popup-reg') ||
                event.target.matches('.better-social-share-popup-close') ||
                event.target.matches('.better-social-share-close')
              ) {
                event.preventDefault();
                hidePopup();
                // Original code had resetSearch() here, but it's redundant
                // as hidePopup() already calls it. Removed for cleaner logic.
              }
            },
          );
        },
      );

      // Close Popup on ESC keyup
      // Attach to document once.
      once('social-share-esc-doc', 'body', document).forEach(
        function (bodyElement) {
          $(bodyElement).on('keyup', function (event) {
            if (event.which === 27) {
              hidePopup();
            }
          });
        },
      );

      // ----------------------------------------------------------------------
      // 2. Initial AJAX Call (Runs once per page load)
      // ----------------------------------------------------------------------

      if (!isFunctionExecuted) {
        const ajaxUrl = drupalSettings.base_url;

        $.ajax({
          url: ajaxUrl,
          type: 'GET',
          dataType: 'json',
          success(response) {
            // Append the content to the end of the body
            $('body').append(response.content);
            // Re-attach behaviors to the newly loaded content
            Drupal.attachBehaviors($('body').get(0), drupalSettings);
          },
        });

        // Set the flag to true
        isFunctionExecuted = 1;
      }

      // ----------------------------------------------------------------------
      // 3. Social Link Click Handler (Share Link Generation)
      // ----------------------------------------------------------------------

      once('social-link-click', '.social-link', context).forEach(
        function (linkElement) {
          const $this = $(linkElement);

          $this.on('click', function (e) {
            if (!$this.hasClass('no-popup')) {
              e.preventDefault();

              if (
                $('.better-social-share-popup').hasClass('is-visible') ||
                $this.hasClass('copy-link') ||
                $this.data('link')
              ) {
                let entityUrl = $('.better-social-share-popup-container').data(
                  'entity_url',
                );
                let entityTitle = $(
                  '.better-social-share-popup-container',
                ).data('entity_title');

                if ($this.data('link')) {
                  const dataLink = $this.data('link');

                  if (entityUrl === '') {
                    entityUrl =
                      drupalSettings.current_url || window.location.href;
                    entityTitle =
                      drupalSettings.current_title || document.title;
                  }

                  const replacedUrl = dataLink
                    .replace('$share_link', encodeURIComponent(entityUrl))
                    .replace('$title', encodeURIComponent(entityTitle));

                  window.open(
                    replacedUrl,
                    'popUpWindow',
                    'height=400,width=600,left=400,top=100,resizable,scrollbars,toolbar=0,personalbar=0,menubar=no,location=no,directories=no,status',
                  );
                  hidePopup();
                } else if ($this.hasClass('copy-link')) {
                  // MODERN CLIPBOARD API
                  if (navigator.clipboard) {
                    navigator.clipboard
                      .writeText(entityUrl)
                      .then(() => {
                        alert(Drupal.t('Link copied to clipboard!'));
                      })
                      .catch((err) => {
                        console.error('Could not copy text: ', err);
                        alert(
                          Drupal.t(
                            'Failed to copy link. Please copy manually.',
                          ),
                        );
                      });
                  } else {
                    // Fallback for older browsers
                    const $tempTextarea = $('<textarea>');
                    $tempTextarea.get(0).value = entityUrl;

                    $('body').append($tempTextarea);
                    $tempTextarea.select();
                    document.execCommand('copy');
                    $tempTextarea.remove();
                    alert(Drupal.t('Text copied to clipboard!'));
                  }
                  hidePopup();
                } else if ($this.attr('href')) {
                  e.preventDefault();
                  const dataLink = $this.attr('href');
                  const replacedUrl = dataLink
                    .replace('$share_link', encodeURIComponent(entityUrl))
                    .replace('$title', encodeURIComponent(entityTitle));

                  window.location.href = replacedUrl;
                }
              }
            } else {
              // If the link has .no-popup
              // do nothing
            }
          });
        },
      );
    },
  };

  // --- Global Functions (Outside Drupal.behaviors) ---

  /**
   * Search sharing services.
   * @param {string} val - The search term.
   */
  window.heateorSsspFilterSharing = function (val) {
    const searchTerm = val.toLowerCase();

    // Iterating over the parent li elements.
    $('ul.better-social-share-mini li').each(function () {
      const $this = $(this);
      const linkText = $this.find('a').get(0).textContent.toLowerCase();

      if (linkText.indexOf(searchTerm) !== -1) {
        $this.show();
      } else {
        $this.hide();
      }
    });
  };
})(jQuery, Drupal, drupalSettings);
