/**
 * Copy the current page url to clipboard when clicking on the .btnCopy button.
 * Based on
 * https://stackoverflow.com/questions/400212/how-do-i-copy-to-the-clipboard-in-javascript/30810322#30810322
 */

(function ($) {
  'use strict';

  // Store a selector string for future use in finding copy indicator.
  Drupal.better_social_sharing_popup_child_selector = '.social-sharing-buttons__popup';
  // Store classname string for future use in copy indicator visibility toggle.
  Drupal.better_social_sharing_indicator_visible_class = 'visible';

  // Selector of the element that copies link when clicked.
  Drupal.behaviors.copyButtonElements = {
    attach: function (context) {
      var $copyButton = $('.btnCopy');
      $copyButton.once('copy-current-url').each(function () {
        // Adding click event on each anchor element.
        $(this).on('click', function (e) {
          e.preventDefault();
          Drupal.copyTextToClipboard(window.location.href, $(this));
        });
      });
    }
  };

  // Function to copy current url to clipboard.
  // Shows a popupmessage on screen if url was copied successful.
  Drupal.copyTextToClipboard = function (text, $clickedElement) {
    if (!navigator.clipboard) {
      Drupal.fallbackCopyTextToClipboard(text, $clickedElement);
      return;
    }

    navigator.clipboard.writeText(text, $clickedElement)
      .then(function () {
        Drupal.showCopiedMessage($clickedElement);
      }, function (err) {
        console.error('Error copying current url to clipboard: ', err);
      });
  };

  // Fallback copy functionality using using older document.execCommand('copy') for when the normal clipboard
  // functionality (navigator.clipboard) does not work. This generates a textarea with url as content and the copies that
  // content using the document.execCommand('copy') command.
  Drupal.fallbackCopyTextToClipboard = function (text, $clickedElement) {

    var $inputURL = $("<input>");
    $("body").append($inputURL);
    $inputURL.val(text).select();

    try {
      document.execCommand("copy");
      Drupal.showCopiedMessage($clickedElement);
    }
    catch (err) {
      console.error('Error copying current url to clipboard', err);
    }

    $inputURL.remove();
  };

  // Show a popup if the current url was successfully copied.
  Drupal.showCopiedMessage = function ($clickedElement) {
    if (!$clickedElement && !$clickedElement.length) {
      return;
    }

    var $popupElement = $(Drupal.better_social_sharing_popup_child_selector, $clickedElement);

    $popupElement.addClass(Drupal.better_social_sharing_indicator_visible_class);

    setTimeout(function () {
      $popupElement.removeClass(Drupal.better_social_sharing_indicator_visible_class);
    }, 4000);
  };

})(jQuery, Drupal);
