<?php

namespace Drupal\betterembed\Entity;

use Drupal\Component\Utility\Random;
use Drupal\Core\Entity\Entity;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\RevisionableContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Language\Language;
use Drupal\file\Entity\File;
use Drupal\media\Entity\Media;
use Drupal\user\UserInterface;
use GuzzleHttp\Client;

/**
 * Defines the Better Embed Entity entity.
 *
 * @ingroup betterembed
 *
 * @ContentEntityType(
 *   id = "betterembed",
 *   label = @Translation("Better Embed Entity"),
 *   handlers = {
 *     "storage" = "Drupal\betterembed\BetterEmbedEntityStorage",
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\betterembed\BetterEmbedEntityListBuilder",
 *     "views_data" = "Drupal\betterembed\Entity\BetterEmbedEntityViewsData",
 *     "translation" = "Drupal\betterembed\BetterEmbedEntityTranslationHandler",
 *
 *     "form" = {
 *       "default" = "Drupal\betterembed\Form\BetterEmbedEntityForm",
 *       "add" = "Drupal\betterembed\Form\BetterEmbedEntityForm",
 *       "edit" = "Drupal\betterembed\Form\BetterEmbedEntityForm",
 *       "delete" = "Drupal\betterembed\Form\BetterEmbedEntityDeleteForm",
 *     },
 *     "access" = "Drupal\betterembed\BetterEmbedEntityAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\betterembed\BetterEmbedEntityHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "betterembed",
 *   data_table = "betterembed_field_data",
 *   revision_table = "betterembed_revision",
 *   revision_data_table = "betterembed_field_revision",
 *   translatable = TRUE,
 *   admin_permission = "administer better embed entity entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "revision" = "vid",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *     "uid" = "user_id",
 *     "langcode" = "langcode",
 *     "status" = "status",
 *   },
 *   links = {
 *     "canonical" = "/betterembed/{betterembed}",
 *     "add-form" = "/admin/betterembed/add",
 *     "edit-form" = "/admin/betterembed/{betterembed}/edit",
 *     "delete-form" = "/admin/betterembed/{betterembed}/delete",
 *     "version-history" = "/admin/betterembed/{betterembed}/revisions",
 *     "revision" = "/admin/betterembed/{betterembed}/revisions/{betterembed_revision}/view",
 *     "revision_revert" = "/admin/betterembed/{betterembed}/revisions/{betterembed_revision}/revert",
 *     "revision_delete" = "/admin/betterembed/{betterembed}/revisions/{betterembed_revision}/delete",
 *     "translation_revert" = "/admin/betterembed/{betterembed}/revisions/{betterembed_revision}/revert/{langcode}",
 *     "collection" = "/admin/content/betterembed",
 *   },
 *   field_ui_base_route = "betterembed.settings"
 * )
 */
class BetterEmbedEntity extends RevisionableContentEntityBase implements BetterEmbedEntityInterface {

  use EntityChangedTrait;

  /**
   * Load by URL
   */
  public static function loadByURL($url) {
    $entity_type_repository = \Drupal::service('entity_type.repository');
    $entity_type_manager = \Drupal::entityTypeManager();
    $storage = $entity_type_manager->getStorage($entity_type_repository->getEntityTypeFromClass(get_called_class()));
    return $storage->loadByProperties([ 'name' => $url ]);
  }

  public static function createWithJson($name, $json) {
    $values = [];
    if (!is_array($json)) {
      $values = [
        'name'                => !empty($name)? $name : '',
        'field_embed_html'    => !empty($json->embedHtml)? $json->embedHtml : '',
        'field_url'           => !empty($json->url)? $json->url : '',
        'field_item_type'     => !empty($json->itemType)? $json->itemType : '',
        'field_title'         => !empty($json->title)? $json->title : '',
        'field_body'          => !empty($json->body)? $json->body : '',
        'field_thumbnail_url' => !empty($json->thumbnailUrl)? $json->thumbnailUrl : '',
        'field_author_name'   => !empty($json->authorName)? $json->authorName : '',
        'field_author_url'    => !empty($json->authorUrl)? $json->authorUrl : '',
        'field_response_json'    => json_encode($json),
        // TODO - save in proper format
        //'field_published_at'  => !empty($json->publishedAt)? $json->publishedAt : '',
      ];
    }

    return parent::create($values);
  }

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);
    $values += [
      'user_id' => \Drupal::currentUser()->id(),
    ];
  }


  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage) {
    parent::preSave($storage);

    foreach (array_keys($this->getTranslationLanguages()) as $langcode) {
      $translation = $this->getTranslation($langcode);

      // If no owner has been set explicitly, make the anonymous user the owner.
      if (!$translation->getOwner()) {
        $translation->setOwnerId(0);
      }
    }

    // If no revision author has been set explicitly, make the betterembed owner the
    // revision author.
    if (!$this->getRevisionUser()) {
      $this->setRevisionUserId($this->getOwnerId());
    }

    if (!empty($this->field_thumbnail_url->uri) && empty($this->field_thumbnail->target_id)) {

      $client = new Client(['base_uri' => $this->field_thumbnail_url->uri]);
      $request = $client->request("GET");

      if ($request->getStatusCode() == 200) {
        $image_data = $request->getBody()->getContents();
        $file_type = explode('/', $request->getHeader('Content-Type')[0]);
        $file_type = $file_type[1];

        $directory = file_default_scheme() . '://betterembed';
        file_prepare_directory($directory, FILE_CREATE_DIRECTORY | FILE_MODIFY_PERMISSIONS);
        $random = new Random();
        $file = file_save_data($image_data, $directory . '/thumbnail_ ' . $random->word(5) . '.' . $file_type, FILE_EXISTS_RENAME);
        $file->save();

        $media_entity = Media::create([
          'bundle' => 'image',
          'name' => $this->getName() . ' Thumbnail',
          'field_media_image' => [
            'target_id' => $file->id(),
          ],
        ]);
        $media_entity->save();

        $this->field_thumbnail->target_id = $media_entity->id();
      }
      else {
        $error = $request->getBody()->getContents();
        \Drupal::logger('BetterEmbed')
          ->notice("Embed ID @id: Couldn't get Thumbnail from @url: @error",
                    [ '@id' => $this->id(), '@url' => $this->field_thumbnail_url->uri, '@error' => $error ]);
      }
    }
  }

  function delete() {

    $media_entity = $this->field_thumbnail->entity;

    if ($media_entity) {
      $media_entity->delete();
    }

    parent::delete();
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return $this->get('name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setName($name) {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('user_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('user_id')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('user_id', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('user_id', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isPublished() {
    return (bool) $this->getEntityKey('status');
  }

  /**
   * {@inheritdoc}
   */
  public function setPublished($published) {
    $this->set('status', $published ? TRUE : FALSE);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['user_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The user ID of author of the Better Embed Entity entity.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 5,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name of the Better Embed Entity entity.'))
      ->setRevisionable(TRUE)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Publishing status'))
      ->setDescription(t('A boolean indicating whether the Better Embed Entity is published.'))
      ->setRevisionable(TRUE)
      ->setDefaultValue(TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    $fields['revision_translation_affected'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Revision translation affected'))
      ->setDescription(t('Indicates if the last edit of a translation belongs to current revision.'))
      ->setReadOnly(TRUE)
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE);

    return $fields;
  }

}
