<?php

namespace Drupal\betterembed\Controller;

use Drupal\Component\Utility\Xss;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Url;
use Drupal\betterembed\Entity\BetterEmbedEntityInterface;

/**
 * Class BetterEmbedEntityController.
 *
 *  Returns responses for Better Embed Entity routes.
 */
class BetterEmbedEntityController extends ControllerBase implements ContainerInjectionInterface {

  /**
   * Displays a Better Embed Entity  revision.
   *
   * @param int $betterembed_revision
   *   The Better Embed Entity  revision ID.
   *
   * @return array
   *   An array suitable for drupal_render().
   */
  public function revisionShow($betterembed_revision) {
    $betterembed = $this->entityManager()->getStorage('betterembed')->loadRevision($betterembed_revision);
    $view_builder = $this->entityManager()->getViewBuilder('betterembed');

    return $view_builder->view($betterembed);
  }

  /**
   * Page title callback for a Better Embed Entity  revision.
   *
   * @param int $betterembed_revision
   *   The Better Embed Entity  revision ID.
   *
   * @return string
   *   The page title.
   */
  public function revisionPageTitle($betterembed_revision) {
    $betterembed = $this->entityManager()->getStorage('betterembed')->loadRevision($betterembed_revision);
    return $this->t('Revision of %title from %date', ['%title' => $betterembed->label(), '%date' => format_date($betterembed->getRevisionCreationTime())]);
  }

  /**
   * Generates an overview table of older revisions of a Better Embed Entity .
   *
   * @param \Drupal\betterembed\Entity\BetterEmbedEntityInterface $betterembed
   *   A Better Embed Entity  object.
   *
   * @return array
   *   An array as expected by drupal_render().
   */
  public function revisionOverview(BetterEmbedEntityInterface $betterembed) {
    $account = $this->currentUser();
    $langcode = $betterembed->language()->getId();
    $langname = $betterembed->language()->getName();
    $languages = $betterembed->getTranslationLanguages();
    $has_translations = (count($languages) > 1);
    $betterembed_storage = $this->entityManager()->getStorage('betterembed');

    $build['#title'] = $has_translations ? $this->t('@langname revisions for %title', ['@langname' => $langname, '%title' => $betterembed->label()]) : $this->t('Revisions for %title', ['%title' => $betterembed->label()]);
    $header = [$this->t('Revision'), $this->t('Operations')];

    $revert_permission = (($account->hasPermission("revert all better embed entity revisions") || $account->hasPermission('administer better embed entity entities')));
    $delete_permission = (($account->hasPermission("delete all better embed entity revisions") || $account->hasPermission('administer better embed entity entities')));

    $rows = [];

    $vids = $betterembed_storage->revisionIds($betterembed);

    $latest_revision = TRUE;

    foreach (array_reverse($vids) as $vid) {
      /** @var \Drupal\betterembed\BetterEmbedEntityInterface $revision */
      $revision = $betterembed_storage->loadRevision($vid);
      // Only show revisions that are affected by the language that is being
      // displayed.
      if ($revision->hasTranslation($langcode) && $revision->getTranslation($langcode)->isRevisionTranslationAffected()) {
        $username = [
          '#theme' => 'username',
          '#account' => $revision->getRevisionUser(),
        ];

        // Use revision link to link to revisions that are not active.
        $date = \Drupal::service('date.formatter')->format($revision->getRevisionCreationTime(), 'short');
        if ($vid != $betterembed->getRevisionId()) {
          $link = $this->l($date, new Url('entity.betterembed.revision', ['betterembed' => $betterembed->id(), 'betterembed_revision' => $vid]));
        }
        else {
          $link = $betterembed->link($date);
        }

        $row = [];
        $column = [
          'data' => [
            '#type' => 'inline_template',
            '#template' => '{% trans %}{{ date }} by {{ username }}{% endtrans %}{% if message %}<p class="revision-log">{{ message }}</p>{% endif %}',
            '#context' => [
              'date' => $link,
              'username' => \Drupal::service('renderer')->renderPlain($username),
              'message' => ['#markup' => $revision->getRevisionLogMessage(), '#allowed_tags' => Xss::getHtmlTagList()],
            ],
          ],
        ];
        $row[] = $column;

        if ($latest_revision) {
          $row[] = [
            'data' => [
              '#prefix' => '<em>',
              '#markup' => $this->t('Current revision'),
              '#suffix' => '</em>',
            ],
          ];
          foreach ($row as &$current) {
            $current['class'] = ['revision-current'];
          }
          $latest_revision = FALSE;
        }
        else {
          $links = [];
          if ($revert_permission) {
            $links['revert'] = [
              'title' => $this->t('Revert'),
              'url' => $has_translations ?
              Url::fromRoute('entity.betterembed.translation_revert', ['betterembed' => $betterembed->id(), 'betterembed_revision' => $vid, 'langcode' => $langcode]) :
              Url::fromRoute('entity.betterembed.revision_revert', ['betterembed' => $betterembed->id(), 'betterembed_revision' => $vid]),
            ];
          }

          if ($delete_permission) {
            $links['delete'] = [
              'title' => $this->t('Delete'),
              'url' => Url::fromRoute('entity.betterembed.revision_delete', ['betterembed' => $betterembed->id(), 'betterembed_revision' => $vid]),
            ];
          }

          $row[] = [
            'data' => [
              '#type' => 'operations',
              '#links' => $links,
            ],
          ];
        }

        $rows[] = $row;
      }
    }

    $build['betterembed_revisions_table'] = [
      '#theme' => 'table',
      '#rows' => $rows,
      '#header' => $header,
    ];

    return $build;
  }

}
