<?php
// $Id: file.inc,v 1.1 2005/11/12 18:39:29 ber Exp $

/**
 * @file
 * API for handling file uploads and server file management.
 */
 
/**
 * Return a list of available toolkits.
 *
 * @return An array of toolkit name => descriptive title.
 */
function file_get_available_toolkits() {
  $toolkits = file_scan_directory('includes', 'file\..*\.inc$');

  $output = array();
  foreach ($toolkits as $file => $toolkit) {
    include_once "./$file";
    $function = str_replace('.', '_', $toolkit->name) .'_info';
    if (function_exists($function)) {
      $info = $function();
      variable_set('file_toolkit_file_'.$info['protocol'], $file);
      $output[$info['name']] = $info['title'];
    }
  }
  return $output;
}

/**
 * Retrieve the name of the currently used toolkit.
 *
 * @return String containing the name of the toolkit.
 */
function file_get_toolkit($protocol) {
  $toolkit_file = variable_get('file_toolkit_file_'. $protocol, '');
  if (file_exists($toolkit_file)) {
    include_once $toolkit_file;
    $toolkit = $protocol;
  }
  else {
    $toolkit = false;
  }
  return $toolkit;
}

/**
 * Get the protocol string from a file-path
 * 
 * @param $file a file object
 * @return $file a file object, with $file->path and $file->protocol filled in
 **/
function file_get_protocol($file) {
  list($file->protocol, $file->path) = explode('://', $file->path);
  // Recreate a path for future use. 
  $file->path = file_create_path($file->path);
  return $file;
}

/**
 * Invokes the given method using the currently selected toolkit.
 *
 * @param $file (object). Contains at least $file->path, in the form of file://path/to/file.ext or http://www.example.com/file.ext
 * @param $method A string containing the method to invoke.
 * @param $params An optional array of parameters to pass to the toolkit method.
 *
 * @return Mixed values (typically Boolean for successful operation).
 */
function file_toolkit_invoke($file, $method, $params = array()) {
  $file = file_get_protocol($file);
  array_unshift($params, $file->path);
  if ($toolkit = file_get_toolkit($file->protocol)) {
    $function = 'file_'. $toolkit .'_'. $method;
    if (function_exists($function)) {
      return call_user_func_array($function, $params);
    }
    else {
      watchdog('php', t("The selected file handling toolkit '%toolkit' can not correctly process '%function'.", array('%toolkit' => "<em>$toolkit</em>", '%function' => "<em>$function</em>")), WATCHDOG_ERROR);
      return false;
    }
  }
  else {
      watchdog('php', t('Fatal error: There is no file toolkit available!')), WATCHDOG_ERROR);
      return false;
  }
}

/**
 * Create the download path to a file.
 *
 * @param $path Path to the file to generate URL for
 * @return URL pointing to the file
 */
function file_create_url($path) {
  return file_toolkit_invoke($path, 'create_url');
}

/**
 * Make sure the destination is a complete path and resides in the
 * file system directory, if it is not prepend the
 * file system directory.
 *
 * @param $dest Path to verify
 * @return Path to file with file system directory appended if necessary.
 *         Returns FALSE if the path is invalid (i.e. outside the configured 'files'-directory).
 */
function file_create_path($dest = 0) {
  return file_toolkit_invoke($dest, 'create_path');
}

/**
 * Check that directory exists and is writable.
 *
 * @param $directory Path to extract and verify directory for.
 * @param $mode Try to create the directory if it does not exist.
 * @param $form_item Optional name for a field item to attach potential errors to.
 * @return False when directory not found, or true when directory exists.
 */
function file_check_directory($directory, $mode = 0, $form_item = NULL) {
  return file_toolkit_invoke($dest, 'check_directory');
}

/**
 * Checks path to see if it is a directory, or a dir/file.
 *
 * @param $path
 */
function file_check_path($path) {
   //TODO should check for protocols in paths too
   return file_toolkit_invoke($dest, 'check_path');
}

/**
 * Check if $source is a valid file upload.
 *
 * @param $source Can be a file object, or a filename (string).
 * @return Returns a file object
 */
function file_check_upload($source) {
  return file_toolkit_invoke($dest, 'check_upload');
}

/**
 * Check if a file is really located inside $directory. Should be used to make
 * sure a file specified is really located within the directory to prevent
 * exploits.
 *
 * @code
 *   // Returns false:
 *   file_check_location('/www/example.com/files/../../../etc/passwd', '/www/example.com/files');
 * @endcode
 *
 * @param $source A string set to the file to check.
 * @param $directory A string where the file should be located.
 * @return 0 for invalid path or the real path of the source.
 */
function file_check_location($source, $directory = 0) {
  $check = realpath($source);
  if ($check) {
    $source = $check;
  }
  else {
    // This file does not yet exist
    $source = realpath(dirname($source)) .'/'. basename($source);
  }
  $directory = realpath($directory);
  if ($directory && strpos($source, $directory) !== 0) {
    return 0;
  }
  return $source;
}

/**
 * Copies a file to a new location. This is a powerful function that in many ways
 * performs like an advanced version of copy().
 * - Checks if $source and $dest are valid and readable/writable.
 * - Performs a file copy if $source is not equal to $dest.
 * - If file already exists in $dest either the call will error out, replace the
 *   file or rename the file based on the $replace parameter.
 *
 * @param $source A string specifying the file location of the original file.
 *   This parameter will contain the resulting destination filename in case of
 *   success.
 * @param $dest A string containing the directory $source should be copied to.
 * @param $replace Replace behavior when the destination file already exists.
 *   - FILE_EXISTS_REPLACE - Replace the existing file
 *   - FILE_EXISTS_RENAME - Append _{incrementing number} until the filename is unique
 *   - FILE_EXISTS_ERROR - Do nothing and return false.
 * @return True for success, false for failure.
 */
function file_copy(&$source, $dest = 0, $replace = FILE_EXISTS_RENAME) {
  $dest = file_create_path($dest);

  $directory = $dest;
  $basename = file_check_path($directory);

  // Make sure we at least have a valid directory.
  if ($basename === false) {
    drupal_set_message(t('The selected file %file could not be uploaded, because the destination %directory is not properly configured.', array('%file' => theme('placeholder', $source), '%directory' => theme('placeholder', $dest))), 'error');
    watchdog('file system', t('The selected file %file could not not be uploaded, because the destination %directory could not be found, or because its permissions do not allow the file to be written.', array('%file' => theme('placeholder', $source), '%directory' => theme('placeholder', $dest))), WATCHDOG_ERROR);
    return 0;
  }

  // Process a file upload object.
  if (is_object($source)) {
    $file = $source;
    $source = $file->filepath;
    if (!$basename) {
      $basename = $file->filename;
    }
  }

  $source = realpath($source);
  if (!file_exists($source)) {
    drupal_set_message(t('The selected file %file could not be copied, because no file by that name exists.  Please check that you supplied the correct filename.', array('%file' => theme('placeholder', $source))), 'error');
    return 0;
  }

  // If the destination file is not specified then use the filename of the source file.
  $basename = $basename ? $basename : basename($source);
  $dest = $directory .'/'. $basename;

  // Make sure source and destination filenames are not the same, makes no sense
  // to copy it if they are. In fact copying the file will most likely result in
  // a 0 byte file. Which is bad. Real bad.
  if ($source != realpath($dest)) {
    if (file_exists($dest)) {
      switch ($replace) {
        case FILE_EXISTS_RENAME:
          // Destination file already exists and we can't replace is so we try and
          // and find a new filename.
          if ($pos = strrpos($basename, '.')) {
            $name = substr($basename, 0, $pos);
            $ext = substr($basename, $pos);
          }
          else {
            $name = $basename;
          }

          $counter = 0;
          do {
            $dest = $directory .'/'. $name .'_'. $counter++ . $ext;
          } while (file_exists($dest));
          break;

        case FILE_EXISTS_ERROR:
          drupal_set_message(t('The selected file %file could not be copied, because a file by that name already exists in the destination.', array('%file' => theme('placeholder', $source))), 'error');
          return 0;
      }
    }

    if (!@copy($source, $dest)) {
      drupal_set_message(t('The selected file %file could not be copied.', array('%file' => theme('placeholder', $source))), 'error');
      return 0;
    }
  }

  if (is_object($file)) {
    $file->filename = $basename;
    $file->filepath = $dest;
    $source = $file;
  }
  else {
    $source = $dest;
  }

  return 1; // Everything went ok.
}

/**
 * Moves a file to a new location.
 * - Checks if $source and $dest are valid and readable/writable.
 * - Performs a file move if $source is not equal to $dest.
 * - If file already exists in $dest either the call will error out, replace the
 *   file or rename the file based on the $replace parameter.
 *
 * @param $source A string specifying the file location of the original file.
 *   This parameter will contain the resulting destination filename in case of
 *   success.
 * @param $dest A string containing the directory $source should be copied to.
 * @param $replace Replace behavior when the destination file already exists.
 *   - FILE_EXISTS_REPLACE - Replace the existing file
 *   - FILE_EXISTS_RENAME - Append _{incrementing number} until the filename is unique
 *   - FILE_EXISTS_ERROR - Do nothing and return false.
 * @return True for success, false for failure.
 */
function file_move(&$source, $dest = 0, $replace = FILE_EXISTS_RENAME) {

  $path_original = is_object($source) ? $source->filepath : $source;

  if (file_copy($source, $dest, $replace)) {
    $path_current = is_object($source) ? $source->filepath : $source;

    if ($path_original == $path_current || file_delete($path_original)) {
      return 1;
    }
    drupal_set_message(t('The removal of the original file %file has failed.', array('%file' => theme('placeholder', $source))), 'error');
  }
  return 0;
}

function file_create_filename($basename, $directory) {
  $dest = $directory .'/'. $basename;

  if (file_exists($dest)) {
    // Destination file already exists, generate an alternative.
    if ($pos = strrpos($basename, '.')) {
      $name = substr($basename, 0, $pos);
      $ext = substr($basename, $pos);
    }
    else {
      $name = $basename;
    }

    $counter = 0;
    do {
      $dest = $directory .'/'. $name .'_'. $counter++ . $ext;
    } while (file_exists($dest));
  }

  return $dest;
}

function file_delete($path) {
  if (is_file($path)) {
    return unlink($path);
  }
}

/**
 * Saves a file upload to a new location. The source file is validated as a
 * proper upload and handled as such.
 *
 * @param $source A string specifying the name of the upload field to save.
 *   This parameter will contain the resulting destination filename in case of
 *   success.
 * @param $dest A string containing the directory $source should be copied to,
 *   will use the temporary directory in case no other value is set.
 * @param $replace A boolean, set to true if the destination should be replaced
 *   when in use, but when false append a _X to the filename.
 * @return An object containing file info or 0 in case of error.
 */
function file_save_upload($source, $dest = 0, $replace = FILE_EXISTS_RENAME) {
  // Make sure $source exists in $_FILES.
  if ($file = file_check_upload($source)) {
    if (!$dest) {
      $dest = variable_get('file_directory_temp', FILE_DIRECTORY_TEMP);
      $temporary = 1;
      if (is_file($file->filepath)) {
        // If this file was uploaded by this user before replace the temporary copy.
        $replace = 1;
      }
    }

    if (!user_access('bypass input data check') && !valid_input_data($file)) {
      watchdog('security', t('The file %file has not been saved, because it may contain a possible attempt to exploit or abuse this system.', array('%file' => theme('placeholder', $source))), WATCHDOG_WARNING);
      drupal_set_message(t('The file %file has not been saved, because it contains invalid data.', array('%file' => theme('placeholder', $source))), 'error');
      return 0;
    }

    // Check for file upload errors.
    switch ($file->error) {
      case 0: // UPLOAD_ERR_OK: File uploaded successfully
        break;
      case 1: // UPLOAD_ERR_INI_SIZE: File size exceeded php.ini value
      case 2: // UPLOAD_ERR_FORM_SIZE: File size exceeded MAX_FILE_SIZE form value
        drupal_set_message(t('The file %file could not be saved, because it exceeds the maximum allowed size for uploads.', array('%file' => theme('placeholder', $source))), 'error');
        return 0;
      case 3: // UPLOAD_ERR_PARTIAL: File was only partially uploaded
      case 4: // UPLOAD_ERR_NO_FILE: No file was uploaded
        drupal_set_message(t('The file %file could not be saved, because the upload did not complete.', array('%file' => theme('placeholder', $source))), 'error');
        return 0;
      default: // Unknown error
        drupal_set_message(t('The file %file could not be saved. An unknown error has occurred.', array('%file' => theme('placeholder', $source))),'error');
        return 0;
    }

    unset($_SESSION['file_uploads'][is_object($source) ? $source->source : $source]);
    if (file_move($file, $dest, $replace)) {
      if ($temporary) {
        $_SESSION['file_uploads'][is_object($source) ? $source->source : $source] = $file;
      }
      return $file;
    }
    return 0;
  }
  return 0;
}

/**
 * Save a string to the specified destination
 *
 * @param $data A string containing the contents of the file
 * @param $dest A string containing the destination location
 *
 * @return A string containing the resulting filename or 0 on error
 */
function file_save_data($data, $dest, $replace = FILE_EXISTS_RENAME) {
  if (!user_access('bypass input data check') && !valid_input_data($data)) {
    watchdog('security', t('The file has not been saved, because it may contain a possible attempt to exploit or abuse this system.'), WATCHDOG_WARNING);
    drupal_set_message(t('The file has not been saved, because it contains invalid data.'), 'error');
    return 0;
  }

  $temp = variable_get('file_directory_temp', FILE_DIRECTORY_TEMP);
  $file = tempnam($temp, 'file');
  if (!$fp = fopen($file, 'wb')) {
    drupal_set_message(t('The file could not be created.'), 'error');
    return 0;
  }
  fwrite($fp, $data);
  fclose($fp);

  if (!file_move($file, $dest, $replace)) {
    return 0;
  }

  return $file;
}

/**
 * Transfer file using http to client. Pipes a file through Drupal to the
 * client.
 *
 * @param $source File to transfer.
 * @param $headers An array of http headers to send along with file.
 */
function file_transfer($source, $headers) {
  ob_end_clean();

  foreach ($headers as $header) {
    header($header);
  }

  $source = file_create_path($source);

  // Transfer file in 1024 byte chunks to save memory usage.
  if ($fd = fopen($source, 'rb')) {
    while (!feof($fd)) {
      print fread($fd, 1024);
    }
    fclose($fd);
  }
  else {
    drupal_not_found();
  }
  exit();
}

/**
 * Call modules to find out if a file is accessible for a given user.
 */
function file_download() {
  $file = $_GET['file'];
  if (file_exists(file_create_path($file))) {
    $list = module_list();
    foreach ($list as $module) {
      $headers = module_invoke($module, 'file_download', $file);
      if ($headers === -1) {
        drupal_access_denied();
      }
      elseif (is_array($headers)) {
        file_transfer($file, $headers);
      }
    }
  }
  drupal_not_found();
}

/**
 * Finds all files that match a given mask in a given
 * directory.
 *
 * @param $dir
 *   The base directory for the scan.
 * @param $mask
 *   The regular expression of the files to find.
 * @param $nomask
 *   An array of files/directories to ignore.
 * @param $callback
 *   The callback function to call for each match.
 * @param $recurse
 *   When TRUE, the directory scan will recurse the entire tree
 *   starting at the provided directory.
 * @param $key
 *   The key to be used for the returned array of files.  Possible
 *   values are "filename", for the path starting with $dir,
 *   "basename", for the basename of the file, and "name" for the name
 *   of the file without an extension.
 * @param $min_depth
 *   Minimum depth of directories to return files from.
 * @param $depth
 *   Current depth of recursion. This parameter is only used internally and should not be passed.
 *
 * @return
 *   An associative array (keyed on the provided key) of objects with
 *   "path", "basename", and "name" members corresponding to the
 *   matching files.
 */
function file_scan_directory($dir, $mask, $nomask = array('.', '..', 'CVS'), $callback = 0, $recurse = TRUE, $key = 'filename', $min_depth = 0, $depth = 0) {
  $key = (in_array($key, array('filename', 'basename', 'name')) ? $key : 'filename');
  $files = array();

  if (is_dir($dir) && $handle = opendir($dir)) {
    while ($file = readdir($handle)) {
      if (!in_array($file, $nomask)) {
        if (is_dir("$dir/$file") && $recurse) {
          $files = array_merge($files, file_scan_directory("$dir/$file", $mask, $nomask, $callback, $recurse, $key, $min_depth, $depth + 1));
        }
        elseif ($depth >= $min_depth && ereg($mask, $file)) {
          $filename = "$dir/$file";
          $basename = basename($file);
          $name = substr($basename, 0, strrpos($basename, '.'));
          $files[$$key] = new stdClass();
          $files[$$key]->filename = $filename;
          $files[$$key]->basename = $basename;
          $files[$$key]->name = $name;
          if ($callback) {
            $callback($filename);
          }
        }
      }
    }

    closedir($handle);
  }

  return $files;
}
