<?php

namespace Drupal\bibcite_import_orcid\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\Entity\User;
use Drupal\bibcite_entity\Entity\Contributor;
use Drupal\bibcite_entity\Entity\Reference;

/**
 * Provides a ORCID Import form.
 */
class ImportForm extends FormBase {

  const LOG_CHANNEL = 'bibcite_import_orcid';

  /** @const */
  private static $bibcite_entity_types = [
    "annotation" => "miscellaneous",
    "artistic-performance",
    "book-chapter" => "book_chapter",
    "book-review" => "miscellaneous",
    "book" => "book",
    "conference-abstract" => "conference_proceedings",
    "conference-paper" => "conference_paper",
    "conference-poster" => "miscellaneous",
    "data-set" => "database",
    "dictionary-entry" => "miscellaneous",
    "disclosure" => "miscellaneous",
    "dissertation-thesis" => "thesis",
    "edited-book" => "book",
    "encyclopedia-entry" => "miscellaneous",
    "invention" => "miscellaneous",
    "journal-article" => "journal_article",
    "journal-issue" => "journal",
    "lecture-speech" => "miscellaneous",
    "license" => "miscellaneous",
    "magazine-article" => "magazine_article",
    "manual" => "miscellaneous",
    "newsletter-article" => "miscellaneous",
    "newspaper-article" => "newspaper_article",
    "online-resource" => "website",
    "other" => "miscellaneous",
    "patent" => "patent",
    "physical-object" => "miscellaneous",
    "preprint" => "unpublished",
    "registered-copyright" => "miscellaneous",
    "report" => "report",
    "research-technique" => "miscellaneous",
    "research-tool" => "miscellaneous",
    "software" => "software",
    "spin-off-company" => "miscellaneous",
    "standards-and-policy" => "miscellaneous",
    "supervised-student-publication" => "miscellaneous",
    "technical-standard" => "miscellaneous",
    "test" => "miscellaneous",
    "trademark" => "miscellaneous",
    "translation" => "miscellaneous",
    "website" => "website",
    "working-paper" => "miscellaneous",
    "undefined" => "miscellaneous",
  ];

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'bibcite_import_orcid_import';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $uid = \Drupal::currentUser()->id();

    $user = User::load($uid);

    $orcid = $user->get('field_orcid')->value;

    $all_works = $this->getAllWorksFromOrcid($orcid);

    $form['works'] = [
      "#tree" => TRUE,
      "#prefix" => '<div class="container">',
      "#suffix" => '</div>',
    ];

    foreach($all_works as $key=>$work){
      $summary = $work['work-summary'][0];
      $title = substr($summary['title']['title']['value'], 0, 200);
      $type = self::$bibcite_entity_types[$summary['type']];
      $work_put_code = $summary['put-code'];
      $publication_date = $summary['publication-date']['year']['value'];

      $workData = $this->getOrcidWorkData($work_put_code, $orcid);
      $contributors = $workData['contributors']['contributor'];
      $description = $workData['short-description'];

      $reference_data = [
        'type' => $type,
        'title' => $title,
        'bibcite_year' => $publication_date,
        'bibcite_abst_e' => $description,
        'work_code' => $work_put_code,
      ];

      $query = \Drupal::entityQuery('bibcite_reference')
      ->condition('title', $title)
      ->condition('type', $type)
      ->execute();

      $options = array();

      if(sizeof($contributors) == 0){
        $person_data = $this->getOrcidPersonData($orcid);

        $options[] = $person_data['name']['given-names']['value'].' '.$person_data['name']['family-name']['value'];
      }else{
        foreach($contributors as $contributor){

          if ($contributor['credit-name']['value'] == NULL){
            $person_data = $this->getOrcidPersonData($orcid);
            $options[] = $person_data['name']['given-names']['value'].' '.$person_data['name']['family-name']['value'];
          }else{
            $options[]  = $contributor['credit-name']['value'];
          }
          
        }
      }  

      if (empty($query)) {

        $element = [];

        $element['work'] = array(
          '#type' => 'fieldset',
          '#title' => t('Title: ' . $title),
        );

        $element['work']['authors'] = [
          '#type' => 'radios',
          '#title' => t('Authors'),
          '#options' => $options,
          '#required' => 'required',
        ];

        $element['reference_works'][] =[
          '#type' => 'textarea',
          '#value' => json_encode($reference_data),
          '#attributes' => [
            'class' => ['hidden'],
          ],
        ];

        $form['works'][] = $element;
      }else if (!empty($query)){

        $bibcite_reference = Reference::load(reset($query));

        $contributors_ids = array_map(function ($referenced_contributor) {
            return $referenced_contributor->id();
        }, $bibcite_reference->author->referencedEntities());

        $existing_contributor_user_ids = \Drupal::entityQuery('user')
                                        ->condition('field_author', $contributors_ids, 'IN')
                                        ->execute();

        if (!in_array($uid, $existing_contributor_user_ids)){
          $element = [];

          $element['work'] = array(
            '#type' => 'fieldset',
            '#title' => t('Title: ' . $title),
          );
  
          $element['reference_works'][] =[
            '#type' => 'textarea',
            '#value' => json_encode($reference_data),
            '#attributes' => [
              'class' => ['hidden'],
            ],
          ];
          
          $element['work']['authors'] = [
            '#type' => 'radios',
            '#title' => t('Authors'),
            '#options' => $options,
            '#required' => 'required',
          ];
  
          foreach ($contributors_ids as $key => $contrib_id){
  
            $query_users = \Drupal::entityQuery('user')
                ->condition('field_author', $contrib_id)
                ->execute();

            $temp = intval(reset($query_users));
  
            if ($uid == $temp) {
              continue;
            }else {
  
              if (is_numeric($temp) && $temp != 0){
  
                $element['work']['authors']['#options_attributes'][$key] = [
                    "disabled" => "disabled",
                ];
              }
            }
          }
  
          $form['works'][] = $element;
        }
      }
    }

    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Send'),
      "#prefix" => '<div class="container">',
      "#suffix" => '</div>',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {

  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    //Get the uid of the current user
    $uid = \Drupal::currentUser()->id();
    //load the information of the user
    $user = User::load($uid);
    //receive all the values from the form_state
    $all_values = $form_state->getValues();
    //filter works from all the values 
    $works_values = $all_values['works'];
    //Receive all the works from form
    $form_works = $form['works'];
    //array with the contributors id
    $current_contributors_in_user = [];
    //array with the selected options of the form
    $selected_option = array();

    foreach ($works_values as $work){
      //insert the author name in the selected option array
      $selected_option[] = $work['work']['authors'];
    }

    foreach($form_works as $key => $element){
      //check if element is an array and is numeric
      if(is_array($element) && is_numeric($key)){
        $disable = array_keys($element['work']['authors']['#options_attributes']);

        //array to receive the options of each work
        $authors_opt = array();
        //insert the option to authors opt
        $authors_opt = $element['work']['authors']['#options'];
        //decode the reference work
        $json_data = json_decode($element['reference_works'][0]['#value'], TRUE);
        //populate reference_data with the content received from de decode
        $reference_data = [
          'type' => $json_data['type'],
          'title' => $json_data['title'],
          'bibcite_year' => $json_data['bibcite_year'],
          'bibcite_abst_e' => $json_data['bibcite_abst_e'],
          'work_code' => $json_data['work_code'],
        ]; 

        $query_reference = \Drupal::entityQuery('bibcite_reference')
                ->condition('title', $json_data['title'])
                ->condition('type', $json_data['type'])
                ->execute();
        
        //convert the selected option from string to numeric
        $opt = intval($selected_option[$key]);

        if ($disable == NULL){
          //create a new reference
          $bibcite_reference = Reference::create($reference_data);
          //create array with contributors ids
          $contributors_ids = array();

          foreach ($authors_opt as $key => $value){
            //create array map with the contributors of especific user
            array_map(function ($contributor) use (&$current_contributors_in_user) {

                $cname = $contributor->first_name[0]->value . " " . $contributor->last_name[0]->value;

                $current_contributors_in_user[$contributor->id()] = $cname;
            }, $user->field_author->referencedEntities());

            ksort($current_contributors_in_user);

            if ($key == $opt){
              $select_author = $value;

              $names = [];

              if (strpos($select_author, ",")) {
                $names = array_reverse(explode(", ", $select_author));
              } else {
                $names = explode(" ", $select_author);
              }

              $cfname = reset($names);
              $clname = ucfirst(end($names));

              $author_name = $cfname . " " . $clname;

              if(empty($current_contributors_in_user)){
                //Create a contributor
                $contributor = Contributor::create([
                  'type' => 'entity_contributor',
                  'last_name' => $clname,
                  'first_name' => $cfname,
                ]);

                $contributor->save();
                //save contibutor id in user field_author
                $user->field_author[] = $contributor->id();

                $user->save();

                $contributors_ids[] = $contributor->id();
          
              }else if(!in_array($author_name , $current_contributors_in_user)){
                //Create a contributor
                $contributor = Contributor::create([
                  'type' => 'entity_contributor',
                  'last_name' => $clname,
                  'first_name' => $cfname,
                ]);

                $contributor->save();
                //save contibutor id in user field_author
                $user->field_author[] = $contributor->id();

                $user->save();

                $contributors_ids[] = $contributor->id();

              }else if(in_array($author_name , $current_contributors_in_user)){
                $contributors_ids[] = array_search($author_name, $current_contributors_in_user);
              }
            }else{
              $names = [];

              if (strpos($value, ",")) {
                $names = array_reverse(explode(", ", $value));
              } else {
                $names = explode(" ", $value);
              }

              $cfname = reset($names);
              $clname = ucfirst(end($names));

              $author_name = $cfname . " " . $clname;

              //Create a contributor
              $contributor = Contributor::create([
                'type' => 'entity_contributor',
                'last_name' => $clname,
                'first_name' => $cfname,
              ]);

              $contributor->save();

              $contributors_ids[] = $contributor->id();
            }

            $bibcite_reference->author->setValue($contributors_ids);
            $bibcite_reference->save();
          }
        }else {
          
          $work_reference = Reference::load(reset($query_reference));

          $current_contributors = [];

          array_map(function ($current_author_entity) use (&$current_contributors) {
              $current_contributors[$current_author_entity->id()] = $current_author_entity->first_name[0]->value . " " . $current_author_entity->last_name[0]->value;
          }, $work_reference->author->referencedEntities());

          $work_contributors = [];

          $keys_contribs = array_keys($current_contributors);

          foreach ($authors_opt as $key => $value){
            
            if ($key == $disable[$key]){
              
              $work_contributors[] = $keys_contribs[$key];
              
            }else if ($key == $opt){
              $select_author = $value;

              $names = [];

              if (strpos($select_author, ",")) {
                $names = array_reverse(explode(", ", $select_author));
              } else {
                $names = explode(" ", $select_author);
              }

              $cfname = reset($names);
              $clname = ucfirst(end($names));

              $author_name = $cfname . " " . $clname;

              if($value == $author_name){
                array_map(function ($contributor) use (&$current_contributors_in_user) {

                    $cname = $contributor->first_name[0]->value . " " . $contributor->last_name[0]->value;
    
                    $current_contributors_in_user[$contributor->id()] = $cname;
                }, $user->field_author->referencedEntities());
    
                ksort($current_contributors_in_user);

                if(empty($current_contributors_in_user)){
                  //Create a contributor
                  $contributor = Contributor::create([
                    'type' => 'entity_contributor',
                    'last_name' => $clname,
                    'first_name' => $cfname,
                  ]);
  
                  $contributor->save();
                  //save contibutor id in user field_author
                  $user->field_author[] = $contributor->id();
  
                  $user->save();
  
                  $work_contributors[] = $contributor->id();
            
                }else if(!in_array($author_name , $current_contributors_in_user)){
                  //Create a contributor
                  $contributor = Contributor::create([
                    'type' => 'entity_contributor',
                    'last_name' => $clname,
                    'first_name' => $cfname,
                  ]);
  
                  $contributor->save();
                  //save contibutor id in user field_author
                  $user->field_author[] = $contributor->id();
  
                  $user->save();
  
                  $work_contributors[] = $contributor->id();
  
                }else if(in_array($author_name , $current_contributors_in_user)){
                  $author_id = array_search($author_name, $current_contributors_in_user);

                  $work_contributors[] = $author_id;
                }
              }else{
                array_map(function ($contributor) use (&$current_contributors_in_user) {

                    $cname = $contributor->first_name[0]->value . " " . $contributor->last_name[0]->value;
    
                    $current_contributors_in_user[$contributor->id()] = $cname;
                }, $user->field_author->referencedEntities());
    
                ksort($current_contributors_in_user);

                if(empty($current_contributors_in_user)){
                  //Create a contributor
                  $contributor = Contributor::create([
                    'type' => 'entity_contributor',
                    'last_name' => $clname,
                    'first_name' => $cfname,
                  ]);

                  $contributor->save();
                  //save contibutor id in user field_author
                  $user->field_author[] = $contributor->id();

                  $user->save();

                  $work_contributors[] = $contributor->id();
            
                }else if(!in_array($author_name , $current_contributors_in_user)){
                  //Create a contributor
                  $contributor = Contributor::create([
                    'type' => 'entity_contributor',
                    'last_name' => $clname,
                    'first_name' => $cfname,
                  ]);

                  $contributor->save();
                  //save contibutor id in user field_author
                  $user->field_author[] = $contributor->id();

                  $user->save();

                  $work_contributors[] = $contributor->id();

                }else if(in_array($author_name , $current_contributors_in_user)){
                  $author_id = array_search($author_name, $current_contributors_in_user);

                  $work_contributors[] = $author_id;
                }
              }
            }else{
              $work_contributors[] = $keys_contribs[$key];
            }

            
          }

          $work_reference->author->setValue($work_contributors);
          $work_reference->save();
        }
      }
    }

    $form_state->setRedirect('bibcite_import_orcid.success');

  }

  function getAllWorksFromOrcid($orcid){

    //Do something here to get any data.
    \Drupal::logger(self::LOG_CHANNEL)->notice("Fetching ORCID data for user: " . $orcid);
    //set url for user works
    $url = "https://pub.orcid.org/" . $orcid . "/works";
    //Initialize cURL session
    $ch = curl_init();
    //Define cURL options
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 600);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: application/vnd.orcid+json',
    ]);
    //Receive the result of the cURL session
    $curl_result = curl_exec($ch);
    //Check if there were any errors during the curl session
    if (curl_error($ch)) {
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        \Drupal::logger(self::LOG_CHANNEL)->notice("cURL error while fetching ORCID works for user " . $orcid . ": " . curl_error($ch));
        if ($http_code == 504) {
            \Drupal::logger(self::LOG_CHANNEL)->notice("ORCID API Timeout");
        }
    }
    //Close cURL session
    curl_close($ch);

    //Decodes the information to JSON
    $all_works = json_decode($curl_result, true);

    $result = $all_works['group'];
    
    return $result;
  }

  function getOrcidWorkData($work_put_code, $orcid){
    $url = "https://pub.orcid.org/" . $orcid . "/works/" . $work_put_code;

    $ch = curl_init();

    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 600);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: application/vnd.orcid+json',
    ]);

    $curl_result = curl_exec($ch);

    if (curl_error($ch)) {
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        \Drupal::logger(self::LOG_CHANNEL)->notice("cURL error while fetching ORCID works for user " . $orcid . ": " . curl_error($ch));
        if ($http_code == 504) {
          \Drupal::logger(self::LOG_CHANNEL)->notice("ORCID API Timeout");
        }
    }

    curl_close($ch);

    $json_data = json_decode($curl_result, true);

    return $json_data['bulk'][0]['work'];
  }

  function getOrcidPersonData($orcid){
    //Set the url for ORCID user data
    $url = "https://pub.orcid.org/" . $orcid;
    //Initialize cURL session
    $ch = curl_init();
    //Define cURL options
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 600);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: application/vnd.orcid+json',
    ]);
    //Receive the result of the cURL session
    $curl_result = curl_exec($ch);
    //Check if there were any errors during the curl session
    if (curl_error($ch)) {
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        \Drupal::logger(self::LOG_CHANNEL)->notice("cURL error while fetching ORCID works for user " . $orcid . ": " . curl_error($ch));
        if ($http_code == 504) {
          \Drupal::logger(self::LOG_CHANNEL)->notice("ORCID API Timeout");
        }
    }
    //Close cURL session
    curl_close($ch);

    //Decodes the information to JSON
    $json_data = json_decode($curl_result, true);
    //Receive the data from the user
    $person_data = $json_data['person'];

    return $person_data;
  }
}