<?php

namespace Drupal\bibcite_import_orcid\Commands;

use Drush\Commands\DrushCommands;
use Drupal\bibcite_entity\Entity\Contributor;
use Drupal\bibcite_entity\Entity\Reference;
use Drupal\user\Entity\User;

/**
 * A Drush commandfile.
 *
 * In addition to this file, you need a drush.services.yml
 * in root of your module, and a composer.json file that provides the name
 * of the services file to use.
 */
class OrcidImportCommands extends DrushCommands {
  /**
   * Imports BibCite references from ORCID
   *
   * @param string $orcid
   *   Argument provided to the drush command.
   *
   * @command bibcite_import_orcid:import
   * @aliases oii
   * @usage bibcite_import_orcid:import 0000-0003-1819-7289
   *
   */
  public function import($orcid) {
      if(empty($orcid)){
          echo "Missing ORCID id.\n";
      }else{
        $result = \Drupal::service("bibcite_import_orcid.importer")->importOrcidData($orcid);

        print_r($result);
      }
  }

  /**
   * Consolidates User's BibCite Contributors
   *
   * @command bibcite_import_orcid:consolidate
   * @aliases ouc
   * @usage bibcite_import_orcid:consolidate
   *
   */
  public function consolidate() {

    $all_users = \Drupal::entityQuery('user')
                  ->execute();

    drush_print("Found ".count($all_users)." users.");
    $contributor_count = 0;

    foreach($all_users as $uid){
        $user = User::load($uid);

        if (empty($user->field_author->referencedEntities())){
            // no Contributors referenced; create new one

            $unames = explode(" ", $user->field_name[0]->value);

            $ufname = reset($unames);
            $ulname = end($unames);

            $contributor = Contributor::create([
                'type' => 'entity_contributor',
                'last_name' => $ulname,
                'first_name' => $ufname,
            ]);

            $contributor->save();

            $user->field_author[] = $contributor->id();

            $contributor_count++;

            drush_print("Created contributor for ".$user->field_name[0]->value.".");

            $user->save();
        }else{
          // get the latest Contributor ID from the list if there is more than 1
          if (count($user->field_author->referencedEntities()) > 1){
            $current_contributors = [];
            
            array_map(function($contributor) use(&$current_contributors){
              $current_contributors[] = $contributor->id();
            }, $user->field_author->referencedEntities());

            sort($current_contributors);

            $latest_contributor_id = end($current_contributors);

            unset($current_contributors[$latest_contributor_id]);

            foreach($current_contributors as $contributor_id_to_remove){
              $query_references = \Drupal::entityQuery('bibcite_reference')
                                    ->condition('author', $contributor_id_to_remove)
                                    ->execute();

              $update_ref_count = 0;

              foreach($query_references as $ref_id){
                $reference = Reference::load($ref_id);
                            
                $contributors_array = array_map(function($referenced_contributor){
                    return $referenced_contributor->id();
                }, $reference->author->referencedEntities());
            
                if (($key = array_search($contributor_id_to_remove, $contributors_array)) !== false) {
                    unset($contributors_array[$key]);
                }
            
                $contributors_array[] = $latest_contributor_id_from_user;
            
                $reference->author = $contributors_array;
            
                $reference->save();
                $update_ref_count++;
              }

              drush_print("Updated ".$update_ref_count." references from old Contributor.");
              $contributor_to_remove = Contributor::load($contributor_id_to_remove);
              $contributor_to_remove->delete();

              drush_print("Deleted old Contributor ID".$contributor_id_to_remove);
            }

          }
        }
    }

    drush_print("Created ".$contributor_count." contributors.");

  }

  /**
   * Imports ORCID data for every User that has ORCID - future version with Queue worker incoming!
   *
   * @command bibcite_import_orcid:import_all
   * @aliases ouia
   * @usage bibcite_import_orcid:import_all
   *
   */

  public function importAll() {
    $all_users = \Drupal::entityQuery('user')
                    ->condition('field_orcid', "", "<>")
                    ->execute();

    drush_print("Found ".count($all_users)." Users with ORCID.");

    $index = 1;
    foreach($all_users as $user_id){
      drush_print("Processing ".$index."/".count($all_users));
      $user = User::load($user_id);

      $orcid_value = $user->get('field_orcid')->getValue();

      $orcid = $orcid_value[0]['value'];

      $result = \Drupal::service("bibcite_import_orcid.importer")->importOrcidData($orcid);
      $index++;

    }

  }

  /**
   * Deletes all entities from bibcite
   *
   * @command bibcite_import_orcid:delete_all
   * @aliases ouda
   * @usage bibcite_import_orcid:delete_all
   *
   */

  public function deleteAll() {
    

    $entity_query_ref = \Drupal::entityQuery('bibcite_reference');

    $all_references = $entity_query_ref->execute();

    echo "Deleting " . count($all_references) . "\n";

    entity_delete_multiple('bibcite_reference', $all_references);
    echo "Done.\n";

    $entity_query_contrib = \Drupal::entityQuery('bibcite_contributor');

    $all_contributors = $entity_query_contrib->execute();

    
    echo "Deleting " . count($all_contributors) . "\n";

    entity_delete_multiple('bibcite_contributor', $all_contributors);
    echo "Done.\n";

  }

}