<?php

namespace Drupal\bibcite_import_orcid\Services;

use Drupal\bibcite_entity\Entity\Contributor;
use Drupal\bibcite_entity\Entity\Reference;
use Drupal\user\Entity\User;
use RenanBr\BibTexParser\Listener;
use RenanBr\BibTexParser\Parser;

/**
 * Class OrcidImporter.
 */
class OrcidImporter
{

    /**
     * Watchdog channel to be used when logging activity.
     */
    const LOG_CHANNEL = 'bibcite_import_orcid';

    /** @const */
    private static $bibcite_entity_types = [
        "annotation" => "miscellaneous",
        "artistic-performance",
        "book-chapter" => "book_chapter",
        "book-review" => "miscellaneous",
        "book" => "book",
        "conference-abstract" => "conference_proceedings",
        "conference-paper" => "conference_paper",
        "conference-poster" => "miscellaneous",
        "data-set" => "database",
        "dictionary-entry" => "miscellaneous",
        "disclosure" => "miscellaneous",
        "dissertation-thesis" => "thesis",
        "edited-book" => "book",
        "encyclopedia-entry" => "miscellaneous",
        "invention" => "miscellaneous",
        "journal-article" => "journal_article",
        "journal-issue" => "journal",
        "lecture-speech" => "miscellaneous",
        "license" => "miscellaneous",
        "magazine-article" => "magazine_article",
        "manual" => "miscellaneous",
        "newsletter-article" => "miscellaneous",
        "newspaper-article" => "newspaper_article",
        "online-resource" => "website",
        "other" => "miscellaneous",
        "patent" => "patent",
        "physical-object" => "miscellaneous",
        "preprint" => "unpublished",
        "registered-copyright" => "miscellaneous",
        "report" => "report",
        "research-technique" => "miscellaneous",
        "research-tool" => "miscellaneous",
        "software" => "software",
        "spin-off-company" => "miscellaneous",
        "standards-and-policy" => "miscellaneous",
        "supervised-student-publication" => "miscellaneous",
        "technical-standard" => "miscellaneous",
        "test" => "miscellaneous",
        "trademark" => "miscellaneous",
        "translation" => "miscellaneous",
        "website" => "website",
        "working-paper" => "miscellaneous",
        "undefined" => "miscellaneous",
    ];

    /**
     * Import works data from ORCID for a user ($user_orcid)
     *
     * @param $user_orcid user ORCID id
     * @return $decoded_data return the works data from ORCID
     */
    public function importOrcidData($user_orcid)
    {
        //Do something here to get any data.
        \Drupal::logger(self::LOG_CHANNEL)->notice("Fetching ORCID data for user: " . $user_orcid);
        //set url for user works
        $url = "https://pub.orcid.org/" . $user_orcid . "/works";
        //Initialize cURL session
        $ch = curl_init();
        //Define cURL options
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 600);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Accept: application/vnd.orcid+json',
        ]);
        //Receive the result of the cURL session
        $curl_result = curl_exec($ch);
        //Check if there were any errors during the curl session
        if (curl_error($ch)) {
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            \Drupal::logger(self::LOG_CHANNEL)->notice("cURL error while fetching ORCID works for user " . $user_orcid . ": " . curl_error($ch));
            if ($http_code == 504) {
                \Drupal::logger(self::LOG_CHANNEL)->notice("ORCID API Timeout");
            }
        }
        //Close cURL session
        curl_close($ch);
        //Call function to receive ORCID works data to populate Bibcite Entities
        $decoded_data = $this->decodeOrcidData($curl_result, $user_orcid);

        return $decoded_data;
    }

    /**
     * Decode ORCID works data to populate Bibcite Entities
     *
     * @param $curl_result transfer the works data from ORCID
     * @return $result
     */
    public function decodeOrcidData($curl_result, $user_orcid)
    {
        //Decodes the information to JSON
        $json_data = json_decode($curl_result, true);
        //Populates Bibcite Entities
        $result = $this->processBibciteEntities($json_data, $user_orcid);

        return $result;
    }

    /**
     * Populate Bibcite Entities
     *
     * @param $json_data
     * @param $orcid
     * @return $result_process
     */
    public function processBibciteEntities($json_data, $orcid)
    {
        //Receive all works from a user
        $all_works = $json_data['group'];
        //Call function to receive ORCID user personal data
        $person_data = $this->getOrcidPersonData($orcid);
        //Search for the user with the $orcid
        $user_query = \Drupal::entityQuery('user')
            ->condition('field_orcid', $orcid)
            ->execute();
        //Receive the first array element
        $user_id = reset($user_query);

        //Load user drupal data
        $user = User::load($user_id);

        //Check if no Contributors referenced and create a new one
        if (empty($user->field_author->referencedEntities())) {
            $user = $this->createNewContributorReference($user);
        }

        $result_process = $this->referenceWorksToAuthors($all_works, $orcid, $user_id);

        return $result_process;
    }

    /**
     * Reference Works to Authors
     *
     * @param $all_works
     * @param $orcid
     * @param $user_id
     * @return $result
     */
    public function referenceWorksToAuthors($all_works, $orcid, $user_id){
        $updated_references_count = 0;
        $created_references_count = 0;
        $referenced_entities = [];
        $contributor_references = [];

        foreach ($all_works as $work) {
            $summary = $work['work-summary'][0];
            $title = substr($summary['title']['title']['value'], 0, 200);
            $type = self::$bibcite_entity_types[$summary['type']];
            $work_put_code = $summary['put-code'];

            $query = \Drupal::entityQuery('bibcite_reference')
                ->condition('title', $title)
                ->condition('type', $type)
                ->execute();

            //Receive authors name for a specific work
            $authors_names = $this->getOrcidWorkContributorNames($work_put_code, $orcid);
            if (!empty($query)) {
                // update work reference authors
                $referenced_entities = $this->updateWorkReferenceAuthors($query, $authors_names, $user_id, $referenced_entities);

                $updated_references_count++;
            } else {
                // create work reference with authors

                $contributor_references = $this->processContributorReferences($authors_names, $user_id);

                $work_data = $this->getOrcidWorkData($work_put_code, $orcid);

                $work_ext_references = $this->getOrcidWorkExtReferences($work_data);

                $reference_data = [
                    'type' => $type,
                    'title' => $title,
                    'bibcite_year' => $work_data['publication-date']['year']['value'],
                    'bibcite_abst_e' => $work_data['short-description'],
                ];

                $referenced_entities = $this->createWorkReferenceWithAuthors($work_data, $work_ext_references, $reference_data, $contributor_references);
                $created_references_count++;
                
            }

        }

        $result = [
            "updated" => $updated_references_count,
            "created" => $created_references_count,
            "entities" => $referenced_entities,
        ];

        return $result;
    }

    /**
     * Create work reference with authors
     *
     * @param $work_data
     * @param $work_ext_references
     * @param $reference_data
     * @param $contributor_references
     * @return $referenced_entities
     */
    public function createWorkReferenceWithAuthors($work_data, $work_ext_references, $reference_data, $contributor_references){
        // handle bibtex if it exists
        if ($work_data['citation'] && $work_data['citation']['citation-type'] == 'BIBTEX') {
            $parser = new Parser();
            $listener = new Listener();

            $parser->addListener($listener);
            $parser->parseString($work_data['citation']['citation-value']);

            $entries = $listener->export();

            foreach ($entries[0] as $prop => $value) {
                switch ($prop) {
                    case 'pages':
                    case 'Pages':
                        $reference_data['bibcite_pages'] = $entries[0][$prop];
                        break;
                    case 'number':
                    case 'Number':
                        $reference_data['bibcite_issue'] = $entries[0][$prop];
                        break;
                    case 'volume':
                    case 'Volume':
                        $reference_data['bibcite_volume'] = $entries[0][$prop];
                        break;
                }
            }
        }

        // add secondary title if available
        if ($work_data['journal-title']) {
            $reference_data['bibcite_secondary_title'] = $work_data['journal-title']['value'];
        }

        // add more references if available
        foreach ($work_ext_references as $ext_ref) {
            switch ($ext_ref['type']) {
                case 'doi':
                    $reference_data['bibcite_url'] = $ext_ref['url'];
                    $reference_data['bibcite_doi'] = $ext_ref['value'];
                    break;
                case 'issn':
                    $reference_data['bibcite_issn'] = $ext_ref['value'];
                    break;
                case 'isbn':
                    $reference_data['bibcite_isbn'] = $ext_ref['value'];
                    break;
                default:
                    $reference_data['bibcite_other_number'] = $ext_ref['value'];
                    break;
            }
        }

        $bibcite_reference = Reference::create($reference_data);

        //$bibcite_reference->author->setValue([]);
        $bibcite_reference->author->setValue($contributor_references);

        $bibcite_reference->save();

        $referenced_entities[] = $bibcite_reference->id();

        return $referenced_entities;
    }

    /**
     * Update work reference authors
     *
     * @param $query
     * @param $authors_names
     * @param $user_id
     * @param $referenced_entities
     * @return $referenced_entities
     */
    public function updateWorkReferenceAuthors($query, $authors_names, $user_id, $referenced_entities){
        $bibcite_reference = Reference::load(reset($query));

        $current_contributors = [];

        if (reset($query) == "3583") {
            $teste = $bibcite_reference->author->getValue();
        }

        array_map(function ($current_author_entity) use (&$current_contributors) {
            $current_contributors[$current_author_entity->id()] = $current_author_entity->first_name[0]->value . " " . $current_author_entity->last_name[0]->value;
        }, $bibcite_reference->author->referencedEntities());

        $contributor_references = $this->processExistingContributorReferences($authors_names, $user_id, $current_contributors);

        $bibcite_reference->author->setValue([]);
        $bibcite_reference->author->setValue($contributor_references);

        $bibcite_reference->save();

        $referenced_entities[] = $bibcite_reference->id();

        return $referenced_entities;
    }


    /**
     * Check if no Contributors referenced and create a new one
     *
     * @param $user
     * @return $user
     */
    public function createNewContributorReference($user){
        //Split the user name with " "
        $unames = explode(" ", $user->field_name[0]->value);
        //Receive the first array element
        $ufname = reset($unames);
        //Receive the last array element
        $ulname = end($unames);
        //Create a contributor
        $contributor = Contributor::create([
            'type' => 'entity_contributor',
            'last_name' => $ulname,
            'first_name' => $ufname,
        ]);

        $contributor->save();
        //save contibutor id in user field_author
        $user->field_author[] = $contributor->id();
        //Increment contibutors count
        $contributor_count++;

        $user->save();
        return $user;
    }

    /**
     * Get personal data from ORCID user ($orcid)
     *
     * @param $orcid ORCID user id
     * @return $decoded_user_data return the personal data from ORCID
     */
    public function getOrcidPersonData($orcid)
    {
        //Set the url for ORCID user data
        $url = "https://pub.orcid.org/" . $orcid;
        //Initialize cURL session
        $ch = curl_init();
        //Define cURL options
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 600);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Accept: application/vnd.orcid+json',
        ]);
        //Receive the result of the cURL session
        $curl_result = curl_exec($ch);
        //Check if there were any errors during the curl session
        if (curl_error($ch)) {
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            \Drupal::logger(self::LOG_CHANNEL)->notice("cURL error while fetching ORCID works for user " . $orcid . ": " . curl_error($ch));
            if ($http_code == 504) {
                \Drupal::logger(self::LOG_CHANNEL)->notice("ORCID API Timeout");
            }
        }
        //Close cURL session
        curl_close($ch);

        //Call function to receive ORCID personal data
        $decoded_user_data = $this->decodeOrcidPersonalData($curl_result);

        return $decoded_user_data;
    }

    /**
     * Decode ORCID personal data from a user
     *
     * @param $curl_result transfer the data from ORCID
     * @return $person_data
     */
    public function decodeOrcidPersonalData($curl_result)
    {
        //Decodes the information to JSON
        $json_data = json_decode($curl_result, true);
        //Receive the data from the user
        $person_data = $json_data['person'];

        return $person_data;
    }

    /**
     * Get ORCID work contributors names
     *
     * @param $work_put_code work id
     * @param $user_orcid ORCID user id
     * @return
     */
    public function getOrcidWorkContributorNames($work_put_code, $user_orcid)
    {
        //Set the url for ORCID work id
        $url = "https://pub.orcid.org/" . $user_orcid . "/works/" . $work_put_code;
        //Initialize cURL session
        $ch = curl_init();
        //Define cURL options
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 600);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Accept: application/vnd.orcid+json',
        ]);
        //Receive the result of the cURL session
        $curl_result = curl_exec($ch);
        //Check if there were any errors during the curl session
        if (curl_error($ch)) {
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            \Drupal::logger(self::LOG_CHANNEL)->notice("cURL error while fetching ORCID works for user " . $user_orcid . ": " . curl_error($ch));
            if ($http_code == 504) {
                \Drupal::logger(self::LOG_CHANNEL)->notice("ORCID API Timeout");
            }
        }
        //Close cURL session
        curl_close($ch);

        //Call function to receive ORCID work contributors
        $decoded_contributors_data = $this->decodeOrcidWorkContributors($curl_result);

        return $decoded_contributors_data;
    }

    /**
     * Decode ORCID personal data from a user
     *
     * @param $curl_result transfer the data from ORCID
     * @return $person_data
     */
    public function decodeOrcidWorkContributors($curl_result)
    {
        //Decodes the information to JSON
        $json_data = json_decode($curl_result, true);
        //Receive al contributors
        $all_contributors = $json_data['bulk'][0]['work']['contributors']['contributor'];

        $contributors_array = [];
        //Creates a array map for all contributors
        $contributors_array = array_map(function ($item) {
            return $item['credit-name']['value'];
        }, $all_contributors);

        return $contributors_array;
    }

    public function processContributorReferences($authors_names, $user_id)
    {

        $contributor_reference_ids = [];

        $user = User::load($user_id);

        $current_contributors_in_user = [];

        array_map(function ($contributor) use (&$current_contributors_in_user) {

            $cname = $contributor->first_name[0]->value . " " . $contributor->last_name[0]->value;

            $current_contributors_in_user[$contributor->id()] = $cname;
        }, $user->field_author->referencedEntities());

        ksort($current_contributors_in_user);

        // if there are no author names coming from Orcid, at least the User's latest Contributor ID must be there

        if (empty($authors_names)) {
            end($current_contributors_in_user);
            return [key($current_contributors_in_user)];
        }

        // there are author strings coming from Orcid
        foreach ($authors_names as $author_str) {

            $names = [];

            if (strpos($author_str, ",")) {
                $names = array_reverse(explode(", ", $author_str));
            } else {
                $names = explode(" ", $author_str);
            }

            $cfname = reset($names);
            $clname = ucfirst(end($names));

            // handle hyphenated last names (poorly...)
            if (strpos($clname, '-')) {
                $clean_lnames = explode("-", $clname);

                $clname = end($clean_lnames);
            }

            $cname = $cfname . " " . $clname;

            $unames = explode(" ", $user->field_name[0]->value);

            $ufname = reset($unames);
            $ulname = end($unames);

            if (in_array($cname, $current_contributors_in_user)) {
                // exact match between author string and User's Contributor name - not very likely but good and simple
                $contributor_reference_ids[] = array_search($cname, $current_contributors_in_user);
                continue;
            } else if ($cfname[0] == $ufname[0] && $clname == $ulname) {
                // first letter of first name and entire last name matches the User's name - very likely this is the Contributor we need
                // use the latest Contributor in the User's profile
                end($current_contributors_in_user);
                $contributor_reference_ids[] = key($current_contributors_in_user);
                continue;
            } else {
                // not from current User; create an anonymous Contributor if the one does not exist
                $contributor = Contributor::create([
                    'type' => 'entity_contributor',
                    'last_name' => $clname,
                    'first_name' => $cfname,
                ]);

                $contributor->save();

                // add the contributor ID to the Reference authors
                $contributor_reference_ids[] = $contributor->id();
                continue;
            }
        }

        $weighted_contributor_reference_ids = [];

        array_map(function ($ref_id) use (&$weighted_contributor_reference_ids) {
            $arr = [
                'target_id' => (string) $ref_id,
                'category' => null,
                'role' => null,
            ];

            $weighted_contributor_reference_ids[] = $arr;

        }, $contributor_reference_ids);

        return $weighted_contributor_reference_ids;
    }

    public function processExistingContributorReferences($authors_names, $user_id, $current_contributors)
    {

        $contributor_reference_ids = [];

        $user = User::load($user_id);

        $current_contributors_in_user =  $this->sortCurrentContributorsInUser($user);

        // if there are no author names coming from Orcid, at least the User's latest Contributor ID must be there

        if (empty($authors_names)) {
            // not enough; needs current Contributors uniqued'
            end($current_contributors_in_user);
            return [key($current_contributors_in_user)];
        }

        // author names not empty, let's process those

        foreach ($authors_names as $author_str) {

            $names = [];

            if (strpos($author_str, ",")) {
                $names = array_reverse(explode(", ", $author_str));
            } else {
                $names = explode(" ", $author_str);
            }

            $cfname = reset($names);
            $clname = ucfirst(end($names));

            // handle hyphenated last names (poorly...)
            if (strpos($clname, '-')) {
                $clean_lnames = explode("-", $clname);

                $clname = end($clean_lnames);
            }

            $cname = $cfname . " " . $clname;

            $unames = explode(" ", $user->field_name[0]->value);

            $ufname = reset($unames);
            $ulname = end($unames);

            // does this author string match any of the current contributors in the Reference?
            //  no  -> create a new anonymous Contributor because it isn't there yet and use that Contributor's ID in the Reference
            //          - also, check if the new Contributor you just created should be referenced by the current User
            //  yes -> is that Contributor anonymous (ie, no User references it)?
            //      yes -> nothing needs to be done, keep using that Contributor's ID in the Reference
            //            - UNLESS it matches the current User. In that case, the Reference should use the User's latest Contributor reference
            //              and the anonymous Contributor should be deleted
            //      no  -> is that Contributor one of the current User's Contributors?
            //          yes -> just make sure the Reference uses the latest Contributor referenced by the current User
            //          no  -> do nothing because it doesn't concern the current User

            $current_contributor_id = $this->isCurrentContributor($cname, $current_contributors);

            if ($current_contributor_id) {
                $matched_contributor_id = $current_contributor_id;

                $query_users = \Drupal::entityQuery('user')
                    ->condition('field_author', $matched_contributor_id)
                    ->execute();

                if (empty($query_users)) {
                    if ($clname == $ulname && $cfname[0] == $ufname[0]) {
                        // matches the current User, so
                        end($current_contributors_in_user);
                        $latest_contributor_id_from_user = key($current_contributors_in_user);

                        // find all References using the matched Contributor ID and replace it with the latest from the User
                        $all_reference_ids = \Drupal::entityQuery('bibcite_reference')
                            ->condition('author', $matched_contributor_id)
                            ->execute();

                        foreach ($all_reference_ids as $ref_id) {
                            $reference = Reference::load($ref_id);

                            $contributors_array = array_map(function ($referenced_contributor) {
                                return $referenced_contributor->id();
                            }, $reference->author->referencedEntities());
                            if (($key = array_search($matched_contributor_id, $contributors_array)) !== false) {
                                unset($contributors_array[$key]);
                            }

                            $contributors_array[] = $latest_contributor_id_from_user;

                            $reference->author = $contributors_array;

                            $reference->save();
                        }

                        $contributor_to_remove = Contributor::load($matched_contributor_id);
                        $contributor_to_remove->delete();

                        $contributor_reference_ids[] = $latest_contributor_id_from_user;

                    } else {
                        $contributor_reference_ids[] = $matched_contributor_id;
                    }
                } else {
                    if (in_array($matched_contributor_id, array_keys($current_contributors_in_user))) {
                        end($current_contributors_in_user);
                        if (array_search($matched_contributor_id, array_keys($current_contributors_in_user)) != key($current_contributors_in_user)) {
                            // the matched Contributor isn't the User's latest Contributor ID, update References as needed

                            $latest_contributor_id_from_user = key($current_contributors_in_user);

                            // find all References using the matched Contributor ID and replace it with the latest from the User
                            $all_reference_ids = \Drupal::entityQuery('bibcite_reference')
                                ->condition('author', $matched_contributor_id)
                                ->execute();

                            foreach ($all_reference_ids as $ref_id) {
                                $reference = Reference::load($ref_id);

                                $contributors_array = array_map(function ($referenced_contributor) {
                                    return $referenced_contributor->id();
                                }, $reference->author->referencedEntities());

                                if (($key = array_search($matched_contributor_id, $contributors_array)) !== false) {
                                    unset($contributors_array[$key]);
                                }

                                $contributors_array[] = $latest_contributor_id_from_user;

                                $reference->author = $contributors_array;

                                $reference->save();
                            }

                            // also, update the current Reference's Contributor IDs to use the latest
                            $contributor_reference_ids[] = $latest_contributor_id_from_user;
                        } else {
                            // already the latest, don't touch
                            $contributor_reference_ids[] = $matched_contributor_id;
                        }
                    } else {
                        // already the latest, don't touch
                        $contributor_reference_ids[] = $matched_contributor_id;
                    }
                }

            } else {
                // does it match the current User?
                if ($clname == $ulname && $cfname[0] == $ufname[0]) {
                    if (!empty($current_contributors_in_user)) {
                        // use the latest Contributor ID referenced by the User if it exists
                        end($current_contributors_in_user);
                        $contributor_reference_ids[] = key($current_contributors_in_user);
                    } else {
                        // no Contributors referenced by the User, create a new one
                        $contributor = Contributor::create([
                            'type' => 'entity_contributor',
                            'last_name' => $clname,
                            'first_name' => $cfname,
                        ]);

                        $contributor->save();

                        $current_contributors_in_user[$contributor->id()] = $cname;

                        $user->field_author[] = $contributor->id();

                        $user->save();

                        $contributor_reference_ids[] = $contributor->id();
                    }
                } else {
                    // does it match other Contributor listed in the reference?
                    $matched_contributor = array_filter($current_contributors, function ($current_contributor) use ($clname, $cfname) {
                        $current_contributor_names = explode(" ", $current_contributor);
                        $ccfname = reset($current_contributor_names);
                        $cclname = end($current_contributor_names);

                        return $cfname[0] == $ccfname[0] && $clname == $cclname;
                    });

                    if (!empty($matched_contributor)) {
                        $contributor_reference_ids[] = key($matched_contributor);
                    } else {
                        // create an anonymous Contributor
                        $contributor = Contributor::create([
                            'type' => 'entity_contributor',
                            'last_name' => $clname,
                            'first_name' => $cfname,
                        ]);

                        $contributor->save();

                        $contributor_reference_ids[] = $contributor->id();
                    }

                }

                continue;
            }

        }

        $weighted_contributor_reference_ids = [];

        array_map(function ($ref_id) use (&$weighted_contributor_reference_ids) {
            $arr = [
                'target_id' => (string) $ref_id,
                'category' => null,
                'role' => null,
            ];

            $weighted_contributor_reference_ids[] = $arr;

        }, $contributor_reference_ids);

        return $weighted_contributor_reference_ids;

    }

    /**
     * Sort current contributors for a user
     *
     * @param $user
     * @return $result
     */
    public function sortCurrentContributorsInUser($user){
        $current_contributors_in_user = [];

        array_map(function ($contributor) use (&$current_contributors_in_user) {

            $cname = $contributor->first_name[0]->value . " " . $contributor->last_name[0]->value;

            $current_contributors_in_user[$contributor->id()] = $cname;
        }, $user->field_author->referencedEntities());

        $result = ksort($current_contributors_in_user);

        return $result;
    }

    /**
     * Get work data from ORCID work code
     *
     * @param $work_put_code word code reference
     * @param $orcid ORCID user id
     * @return $json_data['bulk'][0]['work'] return the information for a specific work
     */
    public function getOrcidWorkData($work_put_code, $orcid)
    {
        $url = "https://pub.orcid.org/" . $orcid . "/works/" . $work_put_code;

        $ch = curl_init();

        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 600);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Accept: application/vnd.orcid+json',
        ]);

        $curl_result = curl_exec($ch);

        if (curl_error($ch)) {
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            \Drupal::logger(self::LOG_CHANNEL)->notice("cURL error while fetching ORCID works for user " . $orcid . ": " . curl_error($ch));
            if ($http_code == 504) {
                \Drupal::logger(self::LOG_CHANNEL)->notice("ORCID API Timeout");
            }
        }

        curl_close($ch);

        $json_data = json_decode($curl_result, true);

        return $json_data['bulk'][0]['work'];
    }

    /**
     * Check if the contributor already exists
     *
     * @param $cname
     * @param $current_contributors
     * @return $result return the contributor id if already exists or false if don't exists
     */
    public function isCurrentContributor($cname, $current_contributors)
    {

        $result = false;

        array_map(function ($current_contributor_id) use (&$cname, &$result, &$current_contributors) {
            $names = [];

            if (strpos($current_contributors[$current_contributor_id], ",")) {
                $names = array_reverse(explode(", ", $current_contributors[$current_contributor_id]));
            } else {
                $names = explode(" ", $current_contributors[$current_contributor_id]);
            }

            $ccfname = reset($names);
            $cclname = ucfirst(end($names));

            $cnames = explode(" ", $cname);

            $cfname = reset($cnames);
            $clname = end($cnames);

            if ($clname == $cclname && $cfname[0] == $ccfname[0]) {
                $result = $current_contributor_id;
            }

        }, array_keys($current_contributors));

        return $result;
    }

    /**
     * Get Orcid external references
     *
     * @param $work_data
     * @return $ext_refs
     */
    public function getOrcidWorkExtReferences($work_data)
    {

        $ext_refs = [];

        foreach ($work_data['external-ids']['external-id'] as $reference) {
            $schema = [
                'type' => $reference['external-id-type'],
                'value' => $reference['external-id-value'],
                'url' => $reference['external-id-url'],
            ];

            $ext_refs[] = $schema;
        }

        return $ext_refs;

    }

    /*public function getOrcidWorkDoi($work_data)
    {
        $doi_data = ['url' => "", "doi" => ""];

        $doi_arr = array_filter($work_data['external-ids']["external-id"], function ($external_id) {
            return $external_id["external-id-type"] == "doi";
        });

        if (!empty($doi_arr)) {
            $doi_data['url'] = $doi_arr[0]["external-id-url"]['value'];
            $doi_data['doi'] = $doi_arr[0]["external-id-value"];
        }

        return $doi_data;

    }*/

    /*public function findDuplicateContributors($user_id)
    {
        $user = User::load($user_id);

        $current_contributor_refs = array_map(function ($contrib_ref) {
            return $contrib_ref->id();
        }, $user->field_author->referencedEntities());

        $contributor_refs_by_name = [];

        foreach ($current_contributor_refs as $contributor_id) {
            $contributor = Contributor::load($contributor_id);

            $cname = $contributor->last_name[0]->value . ", " . $contributor->first_name[0]->value;

            if (!array_key_exists($cname, $contributor_refs_by_name)) {
                $contributor_refs_by_name[$cname] = [];
            }

            $contributor_refs_by_name[$cname][] = $contributor_id;
            ksort($contributor_refs_by_name[$cname]);
        }

        foreach ($contributor_refs_by_name as $contributor_name => $contributor_ids) {
            if (count($contributor_ids) > 1) {
                $oldest_id = $contributor_ids[0];
                unset($contributor_ids[0]);
                $remaining_ids = $contributor_ids;

                foreach ($remaining_ids as $contributor_id_to_remove) {
                    // find references using this contributor
                    $query_references = \Drupal::entityQuery('bibcite_reference')
                        ->condition('author', $contributor_id_to_remove, "CONTAINS")
                        ->execute();

                    // replace with reference to oldest_id

                    foreach ($query_references as $reference_id) {

                        $reference = Reference::load($reference_id);

                        $current_contributors_ids = array_map(function ($cont) {
                            return $cont->id();
                        }, $reference->author->referencedEntities());

                        $final_ids = array_replace($current_contributors_ids, array_fill_keys(array_keys($current_contributors_ids, $contributor_id_to_remove), $oldest_id));

                        $reference->author = $final_ids;

                        $reference->save();
                    }
                    // remove it

                    $storage_handler = \Drupal::entityTypeManager()->getStorage('bibcite_contributor');

                    $contributor_to_remove = $storage_handler->load($contributor_id_to_remove);

                    $storage_handler->delete([$contributor_to_remove]);

                }

            }
        }

        $user->save();

    }*/
}