<?php

namespace Drupal\bible;

/**
 * Interface for Bible file parsers.
 *
 * Defines the contract for parsing Bible files in various formats
 * and preparing them for import into Drupal entities.
 */
interface BibleParserInterface {

  /**
   * Parses a Bible Context file and returns structured data.
   *
   * @param string $uri
   *   The URI of the file to parse.
   *
   * @return array
   *   An array containing parsed Bible data with the following structure:
   *   - 'bible': Array with Bible metadata (shortname, name, langcode, version)
   *   - 'books': Array of book data indexed by book code
   *   - 'verses': Array of verse data grouped by book code and chapter
   *   - 'stats': Array with parsing statistics (line_count, entity_counts)
   *
   * @throws \Drupal\bible\Exception\BibleParseException
   *   Thrown when the file cannot be parsed or contains invalid data.
   */
  public function parseFile(string $uri): array;

  /**
   * Validates a Bible Context file format.
   *
   * @param string $uri
   *   The URI of the file to validate.
   *
   * @return bool
   *   TRUE if the file is valid, FALSE otherwise.
   */
  public function validateFile(string $uri): bool;

  /**
   * Gets supported file extensions.
   *
   * @return array
   *   Array of supported file extensions (without dots).
   */
  public function getSupportedExtensions(): array;

  /**
   * Converts character encoding to UTF-8.
   *
   * @param string $text
   *   The text to convert.
   *
   * @return string
   *   The converted text in UTF-8 encoding.
   */
  public function convertCharacterEncoding(string $text): string;

  /**
   * Prepares batch operations for importing parsed Bible data.
   *
   * @param array $parsed_data
   *   The parsed Bible data array.
   * @param int $chunk_size
   *   The number of items to process per batch operation.
   *
   * @return array
   *   Array of batch operations for processing.
   */
  public function prepareBatchOperations(array $parsed_data, int $chunk_size = 50): array;

  /**
   * Counts the total number of items to be imported.
   *
   * @param array $parsed_data
   *   The parsed Bible data array.
   *
   * @return int
   *   The total number of items to import.
   */
  public function countTotalItems(array $parsed_data): int;

}
