<?php

namespace Drupal\bible\Controller;

use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for Bible reading functionality.
 *
 * @phpstan-consistent-constructor
 */
class BibleController extends ControllerBase {

  /**
   * The previous book in the Bible.
   *
   * @var \Drupal\bible\Entity\BibleBook|null
   */
  protected $previousBook;
  /**
   * The current book being read.
   *
   * @var \Drupal\bible\Entity\BibleBook|null
   */
  protected $currentBook;
  /**
   * The next book in the Bible.
   *
   * @var \Drupal\bible\Entity\BibleBook|null
   */
  protected $nextBook;

  /**
   * The current chapter being read.
   *
   * @var int
   */
  protected $currentChapter;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Creates a new BibleController instance.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The dependency injection container.
   *
   * @return static
   *   A new controller instance.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * Redirects from /bible to the default Bible version.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response to the default Bible reading page.
   */
  public function redirectToDefault(Request $request) {
    $bible = $this->getDefaultBible();

    // Redirect to the default Bible version with Genesis chapter 1.
    return $this->redirect('bible.read', [
      'bible' => $bible->get('shortname')->value,
      'book' => 'GEN',
      'chapter' => 1,
    ]);
  }

  /**
   * Gets the default Bible entity.
   *
   * @return \Drupal\bible\Entity\Bible
   *   The default Bible entity.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   If no Bible versions are available.
   */
  protected function getDefaultBible() {
    // Get the configured default Bible from settings.
    $config = $this->config('bible.settings');
    $defaultBibleId = $config->get('default_bible');

    $bible = NULL;

    // Try to load the configured default Bible.
    if ($defaultBibleId) {
      $bibleStorage = $this->entityTypeManager->getStorage('bible');
      $bible = $bibleStorage->load($defaultBibleId);
    }

    // If no configured default or it doesn't exist, get the first Bible.
    if (!$bible) {
      $bibleStorage = $this->entityTypeManager->getStorage('bible');
      $bibleIds = $bibleStorage->getQuery()
        ->sort('id', 'ASC')
        ->range(0, 1)
        ->accessCheck(FALSE)
        ->execute();

      if ($bibleIds) {
        $bible = $bibleStorage->load(reset($bibleIds));
      }
    }

    // If still no Bible found, show error.
    if (!$bible) {
      throw new NotFoundHttpException('No Bible versions have been imported. Contact your administrator.');
    }

    /** @var \Drupal\bible\Entity\Bible $bible */
    return $bible;
  }

  /**
   * Displays a Bible chapter for reading.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   * @param string $bible
   *   The Bible shortname (e.g., 'KJV').
   * @param string $book
   *   The book code (e.g., 'GEN').
   * @param int $chapter
   *   The chapter number. Defaults to 1.
   *
   * @return array
   *   A render array for the Bible reading interface.
   */
  public function read(Request $request, $bible = 'KJV', $book = 'GEN', $chapter = 1) {
    // Load Bible entity by shortname.
    $bibleEntity = $this->loadBibleByShortname($bible);
    if (!$bibleEntity) {
      throw new NotFoundHttpException('Bible version not found.');
    }

    // Load books and validate current book exists.
    $books = $this->getBooks($bibleEntity->id(), $book);

    if (!$this->currentBook) {
      throw new NotFoundHttpException('Book not found.');
    }

    // Validate chapter number.
    if ($chapter < 1 || $chapter > $this->currentBook->get('chapters')->value) {
      throw new NotFoundHttpException('Chapter not found.');
    }

    $this->currentChapter = $chapter;

    $build = [
      '#theme' => 'bible_read',
      '#books' => $books,
      '#versions' => $this->getVersions(),
      '#current_bible' => $bible,
      '#current_book' => $book,
      '#current_chapter' => $chapter,
      '#verses' => $this->getVerses(),
      '#prev_book' => $this->previousBook?->get('code')->value,
      '#prev_chapter' => $chapter > 1 ? $chapter - 1 : NULL,
      '#next_chapter' => ($chapter < $this->currentBook->get('chapters')->value) ? $chapter + 1 : NULL,
      '#next_book' => $this->nextBook?->get('code')->value,
      '#attached' => [
        'library' => ['bible/bible.read'],
      ],
    ];

    return $build;
  }

  /**
   * Loads a Bible entity by its shortname.
   *
   * @param string $shortname
   *   The Bible shortname (e.g., 'KJV').
   *
   * @return \Drupal\bible\Entity\Bible|null
   *   The Bible entity or null if not found.
   */
  protected function loadBibleByShortname($shortname) {
    $bibleStorage = $this->entityTypeManager->getStorage('bible');
    $bibleIds = $bibleStorage->getQuery()
      ->condition('shortname', $shortname)
      ->accessCheck(FALSE)
      ->execute();

    if (!$bibleIds) {
      return NULL;
    }

    $bibles = $bibleStorage->loadMultiple($bibleIds);
    $bible = reset($bibles);

    /** @var \Drupal\bible\Entity\Bible|null $bible */
    return $bible ?: NULL;
  }

  /**
   * Fetches all books for a given Bible version.
   *
   * @param int $bibleId
   *   The ID of the Bible version to fetch books for. Defaults to 1.
   * @param string $currentBookCode
   *   The code of the current book being read.
   *
   * @return array
   *   An array of BibleBook entities where the key is the book's code and
   *   the value is its name.
   */
  protected function getBooks($bibleId = 1, $currentBookCode = 'GEN') {
    $bookStorage = $this->entityTypeManager->getStorage('bible_book');
    $bookIds = $bookStorage->getQuery()
      ->condition('bible', $bibleId)
      ->sort('id', 'ASC')
      ->accessCheck(FALSE)
      ->execute();
    $books = $bookStorage->loadMultiple($bookIds);
    $bookList = [];
    foreach ($books as $book) {
      /** @var \Drupal\bible\Entity\BibleBook $book */
      $bookList[$book->get('code')->value] = $book->get('name')->value;
      // Logic for which book might be previous/current/next in the list.
      if (!$this->nextBook && $this->currentBook) {
        $this->nextBook = $book;
      }
      if ($book->get('code')->value === $currentBookCode) {
        $this->currentBook = $book;
      }
      if (!$this->currentBook) {
        $this->previousBook = $book;
      }
    }

    return $bookList;
  }

  /**
   * Fetches all available Bible versions.
   *
   * @return array
   *   An array of Bible versions where the key is the shortname and the
   *   value is the full name.
   */
  protected function getVersions() {
    $bibleStorage = $this->entityTypeManager->getStorage('bible');
    $bibleIds = $bibleStorage->getQuery()
      ->sort('id', 'ASC')->accessCheck(FALSE)->execute();
    $bibles = $bibleStorage->loadMultiple($bibleIds);
    $bibleList = [];
    foreach ($bibles as $bible) {
      /** @var \Drupal\bible\Entity\Bible $bible */
      $bibleList[$bible->get('shortname')->value] = $bible->get('name')->value;
    }

    return $bibleList;
  }

  /**
   * Fetches verses for the current book and chapter.
   *
   * @return array
   *   An array of verse data with chapter, verse, and text keys.
   */
  protected function getVerses() {
    $verseStorage = $this->entityTypeManager->getStorage('bible_verse');
    $verseIds = $verseStorage->getQuery()
      ->condition('book', $this->currentBook?->id())
      ->condition('chapter', $this->currentChapter)
      ->sort('id', 'ASC')
      ->accessCheck(FALSE)
      ->execute();
    $verses = $verseStorage->loadMultiple($verseIds);
    $verseList = [];
    foreach ($verses as $verse) {
      /** @var \Drupal\bible\Entity\BibleVerse $verse */
      $verseList[] = [
        'chapter' => $verse->get('chapter')->value,
        'verse' => $verse->get('verse')->value,
        'text' => $verse->get('text')->value,
      ];
    }

    return $verseList;
  }

}
