<?php

namespace Drupal\bible\Exception;

/**
 * Exception thrown when Bible file parsing fails.
 */
class BibleParseException extends \Exception {

  /**
   * The line number where the error occurred.
   *
   * @var int|null
   */
  protected $lineNumber;

  /**
   * The problematic line content.
   *
   * @var string|null
   */
  protected $lineContent;

  /**
   * Creates a new BibleParseException.
   *
   * @param string $message
   *   The exception message.
   * @param int|null $line_number
   *   The line number where the error occurred.
   * @param string|null $line_content
   *   The problematic line content.
   * @param int $code
   *   The exception code.
   * @param \Throwable|null $previous
   *   The previous exception.
   */
  public function __construct(string $message = '', ?int $line_number = NULL, ?string $line_content = NULL, int $code = 0, ?\Throwable $previous = NULL) {
    $this->lineNumber = $line_number;
    $this->lineContent = $line_content;

    if ($line_number !== NULL) {
      $message .= " (Line: {$line_number})";
    }

    if ($line_content !== NULL) {
      $message .= " Content: '{$line_content}'";
    }

    parent::__construct($message, $code, $previous);
  }

  /**
   * Gets the line number where the error occurred.
   *
   * @return int|null
   *   The line number or NULL if not available.
   */
  public function getLineNumber(): ?int {
    return $this->lineNumber;
  }

  /**
   * Gets the problematic line content.
   *
   * @return string|null
   *   The line content or NULL if not available.
   */
  public function getLineContent(): ?string {
    return $this->lineContent;
  }

}
