<?php

namespace Drupal\bible\Form;

use Drupal\bible\BibleParserInterface;
use Drupal\bible\Exception\BibleParseException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileExists;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Http\ClientFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a form for importing Bible translations from various sources.
 *
 * Supports importing from file uploads and GitHub repositories in Bible
 * Context format.
 */
class BibleImportForm extends FormBase {

  /**
   * The HTTP client factory service.
   *
   * @var \Drupal\Core\Http\ClientFactory
   */
  protected $httpClientFactory;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The Bible parser service.
   *
   * @var \Drupal\bible\BibleParserInterface
   */
  protected $bibleParser;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'bible_import';
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new self(
      $container->get('http_client_factory'),
      $container->get('request_stack'),
      $container->get('entity_type.manager'),
      $container->get('bible.parser')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function __construct(ClientFactory $httpClientFactory, RequestStack $requestStack, EntityTypeManagerInterface $entityTypeManager, BibleParserInterface $bibleParser) {
    $this->httpClientFactory = $httpClientFactory;
    $this->requestStack = $requestStack;
    $this->entityTypeManager = $entityTypeManager;
    $this->bibleParser = $bibleParser;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['heading'] = [
      '#markup' => '<h2>' . $this->t('Upload') . '</h2>',
    ];
    $form['description'] = [
      '#type' => 'item',
      '#markup' => $this->t('Upload a .bc or .bc.txt file containing Bible content or select a file to download and install from the list below. You can also upload a Strong Number list with the .sn extension (note: not yet implemented).'),
    ];

    $form['upload'] = [
      '#type' => 'file',
      '#title' => $this->t('File upload'),
      '#description' => $this->t('Upload a file with the extension .bc or .bc.txt or .sn'),
    ];

    // Submit button for file upload.
    $form['submit_upload'] = [
      '#type' => 'submit',
      '#value' => $this->t('Upload'),
      '#button_type' => 'primary',
    ];

    // Add a markup for the heading "Download".
    $form['download_heading'] = [
      '#markup' => '<h2>' . $this->t('Download') . '</h2>',
    ];

    // Add a descriptive item for explaining the download process.
    $form['download_description'] = [
      '#type' => 'item',
      '#markup' => $this->t('Below is a list of files that can be downloaded from GitHub. Clicking "Install" will start the download and installation process for the selected file.'),
    ];

    // Table to download a file from Github.
    $config = $this->config('bible.settings');
    $repo = $config->get('importer.github_repository');
    $client = $this->httpClientFactory->fromOptions([
      'base_uri' => 'https://api.github.com',
    ]);
    $response = $client->get("/repos/{$repo}/contents");
    $files = json_decode($response->getBody());
    $bcFiles = array_filter($files, function ($file) {
      return $file->type === 'file' && substr($file->name, -7) === '.bc.txt';
    });

    $form['files_table'] = [
      '#type' => 'table',
      '#header' => [$this->t('Name'), $this->t('File'), $this->t('Action')],
      '#empty' => $this->t('No files found. Check that the Github repository in configuration is set correctly.'),
    ];

    foreach ($bcFiles as $index => $file) {
      // Loading these names is too expensive an operation for here.
      // @todo Delay this with AJAX and cache it for a long time.
      // $name = $this->fetchAndParseFirstLine($file->download_url);
      $name = "Loading... ({$file->name})";
      $form['files_table'][$index]['name'] = [
        '#markup' => $name,
      ];
      $form['files_table'][$index]['path'] = [
        '#markup' => $file->path,
      ];
      $form['files_table'][$index]['install'] = [
        '#type' => 'submit',
        '#value' => $this->t('Install'),
        '#name' => 'install_' . $index,
        '#submit' => ['::submitDownload'],
        '#attributes' => [
          'download_url' => $file->download_url,
        ],
      ];
    }

    return $form;
  }

  /**
   * Handles form submission for downloading and importing a Bible file.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @todo Move download logic to a separate service.
   */
  public function submitDownload(array &$form, FormStateInterface $form_state) {
    // Find out which button was clicked.
    $triggering_element = $form_state->getTriggeringElement();
    $uri = $triggering_element['#attributes']['download_url'];

    $this->messenger()->addMessage($this->t('Starting download and import: @path', ['@path' => $uri]));

    $this->startBatchImport($uri);

    // Redirect to Bible list after batch completes.
    $form_state->setRedirect('entity.bible.collection');
  }

  /**
   * Fetches and parses the first line of a Bible Context file.
   *
   * @param string $download_url
   *   The download URL of the Bible Context file.
   *
   * @todo Move this to a BC parsed helper class.
   * This is cool but still too slow for 36 files. We could AJAX it
   * and/or cache the results.
   */
  public function fetchAndParseFirstLine($download_url) {
    $context = stream_context_create([
      "http" => [
        "method" => "GET",
    // GitHub requires a user agent.
        "header" => "User-Agent: DrupalBible/1.0",
      ],
    ]);

    $handle = @fopen($download_url, "r", FALSE, $context);

    if ($handle) {
      $lineCount = 0;
      while (($line = fgets($handle)) !== FALSE && $lineCount < 20) {
        // Increment line count so we stop looking after a reasonable number.
        $lineCount++;

        // Strip out Unicode characters from the start of the line.
        $line = preg_replace('/^[\p{Z}\p{C}]+/u', '', $line);

        // Trim the line to remove any whitespace.
        $trimmedLine = trim($line);

        // Skip empty lines or lines starting with ^, *, or #.
        if (empty($trimmedLine) || preg_match('/^[*^#]/', $trimmedLine)) {
          continue;
        }

        // This line is the first non-empty, not starting with ^, *, or #.
        // Close the stream.
        fclose($handle);
        // Return the line.
        return $trimmedLine;
      }
      // Make sure to close the stream if we didn't return early.
      fclose($handle);
    }
    else {
      // Handle error opening the URL.
      return NULL;
    }

    // No suitable line found or couldn't open file.
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Validate file upload.
    $all_files = $this->getRequest()->files->get('files', []);
    if (!empty($all_files['upload'])) {
      $file_upload = $all_files['upload'];
      $extension = $file_upload->getClientOriginalExtension();
      $supported_extensions = $this->bibleParser->getSupportedExtensions();
      // Add Strong Numbers support.
      $supported_extensions[] = 'sn';

      if (!in_array($extension, $supported_extensions)) {
        $form_state->setErrorByName('upload', $this->t('Only files with the following extensions are allowed: @extensions.', [
          '@extensions' => implode(', ', $supported_extensions),
        ]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Handle file upload.
    $request = $this->requestStack->getCurrentRequest();
    $uploaded_file = $request->files->get('files');
    if (!empty($uploaded_file['upload'])) {
      $supported_extensions = $this->bibleParser->getSupportedExtensions();
      // Add Strong Numbers support.
      $supported_extensions[] = 'sn';

      $file = file_save_upload('upload', [
        'file_validate_extensions' => $supported_extensions,
      ], FALSE, 0, FileExists::Replace);

      if ($file) {
        // Store the file so it can be re-used.
        $file->setPermanent();
        $file->save();

        $uri = $file->getFileUri();
        $this->messenger()->addMessage($this->t('File uploaded successfully. Starting import...'));
        $this->startBatchImport($uri);

        // Redirect to Bible list after batch completes.
        $form_state->setRedirect('entity.bible.collection');
      }
      else {
        $this->messenger()->addError($this->t('File upload failed.'));
      }
    }
  }

  /**
   * Starts the batch import process for a Bible file.
   *
   * @param string $uri
   *   The URI of the file to import.
   */
  protected function startBatchImport(string $uri): void {
    try {
      // Validate the file first.
      if (!$this->bibleParser->validateFile($uri)) {
        $this->messenger()->addError($this->t('The file does not appear to be a valid Bible Context file.'));
        return;
      }

      // Parse the file to get structured data.
      $parsed_data = $this->bibleParser->parseFile($uri);

      // Check for parsing errors.
      if (!empty($parsed_data['stats']['errors'])) {
        $this->messenger()->addWarning($this->t('File parsed with @count warnings. Import will continue.', [
          '@count' => count($parsed_data['stats']['errors']),
        ]));

        // Log detailed errors for debugging.
        foreach ($parsed_data['stats']['errors'] as $error) {
          $this->getLogger('bible')->warning('Parse warning on line @line: @error', [
            '@line' => $error['line'],
            '@error' => $error['error'],
          ]);
        }
      }

      // Prepare batch operations.
      // Use larger chunk size for bulk inserts (1000 verses per batch).
      $operations = $this->bibleParser->prepareBatchOperations($parsed_data, 1000);

      $total_items = $this->bibleParser->countTotalItems($parsed_data);

      // Set up the batch.
      $batch = [
        'title' => $this->t('Importing Bible: @name', ['@name' => $parsed_data['bible']['name']]),
        'operations' => $operations,
        'finished' => 'Drupal\\bible\\Service\\BibleBatchOperations::finished',
        'init_message' => $this->t('Starting Bible import...'),
        'progress_message' => $this->t('Processed @current out of @total items.'),
        'error_message' => $this->t('The Bible import has encountered an error.'),
      ];

      batch_set($batch);

      $this->messenger()->addMessage($this->t('Bible import started. Processing @count items...', [
        '@count' => $total_items,
      ]));

    }
    catch (BibleParseException $e) {
      $this->messenger()->addError($this->t('Failed to parse Bible file: @error', [
        '@error' => $e->getMessage(),
      ]));
      $this->getLogger('bible')->error('Bible parse error: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Unexpected error during import: @error', [
        '@error' => $e->getMessage(),
      ]));
      $this->getLogger('bible')->error('Bible import error: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

}
