<?php

namespace Drupal\bible\Plugin\Filter;

use Drupal\filter\Plugin\FilterBase;
use Drupal\filter\FilterProcessResult;
use Drupal\Core\Url;
use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'Bible Reference Filter' filter.
 *
 * @Filter(
 *   id = "bible_filter",
 *   title = @Translation("Bible references as popups"),
 *   type = Drupal\filter\Plugin\FilterInterface::TYPE_TRANSFORM_REVERSIBLE
 * )
 * @phpstan-consistent-constructor
 */
class BibleFilter extends FilterBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a BibleFilter object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, $database, AccountProxyInterface $current_user) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->database = $database;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('database'),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode) {
    $bookNames = $this->loadBooks();

    // Process the text using the extracted book names.
    return $this->doProcess($text, $bookNames);
  }

  /**
   * Return all books available in the database.
   *
   * @return array
   *   Array of book names as strings.
   */
  protected function loadBooks() {
    $query = $this->database->select('bible_book', 'b');
    $query->fields('b', ['name', 'shortname', 'code']);

    $results = $query->execute()->fetchAll();

    $bookNames = [];

    // Populate the book names array with different variations: code, name, and
    // shortname.
    foreach ($results as $book) {
      $bookNames[] = $book->name;
      $bookNames[] = $book->shortname;
      $bookNames[] = $book->code;
      if ($book->code === 'PS') {
        // Hack: many languages skip plural when quoting verses.
        $bookNames[] = 'Psalm';
      }
    }

    // Remove duplicates.
    $bookNames = array_unique($bookNames);

    return $bookNames;
  }

  /**
   * Process text with provided book names for matching Bible references.
   *
   * @param string $text
   *   The text to process.
   * @param array $bookNames
   *   Array of book names to match against.
   *
   * @return \Drupal\filter\FilterProcessResult
   *   The processed result.
   */
  public function doProcess($text, array $bookNames) {
    if (empty($bookNames)) {
      $result = new FilterProcessResult($text);
      return $result;
    }

    // Create a regex pattern to match Bible references including verse ranges.
    $quotedBookNames = array_map(function ($name) {
      return preg_quote($name, '/');
    }, $bookNames);
    $book_pattern = implode('|', $quotedBookNames);
    $pattern = '/\b(' . $book_pattern . ')[\s:]*\d+:\d+(-\d+)?\b/i';

    // Use a callback to replace the matched references with popup content and
    // a link.
    $callback = function ($matches) use ($bookNames) {
      return $this->processReference($matches, $bookNames);
    };

    // Perform the replacement using the regex pattern.
    $new_text = preg_replace_callback($pattern, $callback, $text);

    // Create a FilterProcessResult object and attach the library.
    $result = new FilterProcessResult($new_text);
    $result->addAttachments([
      'library' => [
        'bible/bible.filter',
      ],
    ]);

    return $result;
  }

  /**
   * Process a single Bible reference match.
   *
   * @param array $matches
   *   The regex matches from preg_replace_callback.
   * @param array $bookNames
   *   Array of valid book names.
   *
   * @return string
   *   The processed reference or original text.
   */
  protected function processReference(array $matches, array $bookNames) {
    // Extract the reference.
    $reference = $matches[0];
    $book_name = $matches[1];
    if ($book_name === 'Psalm') {
      $book_name = 'PS';
    }

    // Extract the chapter and verse range from the reference.
    preg_match('/(\d+):(\d+)(-(\d+))?/', $reference, $chapter_verses);
    $chapter = $chapter_verses[1];
    $verse_start = $chapter_verses[2];
    $verse_end = !empty($chapter_verses[4]) ? $chapter_verses[4] : $verse_start;

    // Load the actual book entity to get the code for URL generation.
    $query = $this->entityTypeManager->getStorage('bible_book')->getQuery()
      ->accessCheck(FALSE)
      ->range(0, 1);

    // Create OR condition group for name, shortname, or code matching.
    $or_group = $query->orConditionGroup()
      ->condition('name', $book_name, '=')
      ->condition('shortname', $book_name, '=')
      ->condition('code', $book_name, '=');

    $bookIds = $query->condition($or_group)->execute();

    $books = $this->entityTypeManager->getStorage('bible_book')->loadMultiple($bookIds);
    $book = reset($books);

    if (!$book) {
      return $reference;
    }

    // Query BibleVerse entities for the specified book, chapter, and verse
    // range.
    $verse_ids = $this->entityTypeManager->getStorage('bible_verse')->getQuery()
      ->condition('book', $book->id())
      ->condition('chapter', $chapter)
      ->condition('verse', [$verse_start, $verse_end], 'BETWEEN')
      ->accessCheck(FALSE)
      ->execute();

    // Load the verses and concatenate their text.
    $verses = $this->entityTypeManager->getStorage('bible_verse')->loadMultiple($verse_ids);
    $popup_content = '';
    foreach ($verses as $verse) {
      /** @var \Drupal\bible\Entity\BibleVerse $verse */
      $popup_content .= Html::escape($verse->get('text')->value);
    }

    // Check if user has permission to access the Bible route.
    if ($this->currentUser->hasPermission('view bible entity')) {
      /** @var \Drupal\bible\Entity\BibleBook $book */
      // Generate the link URL using the `bible.read` route and passing the
      // book code and chapter.
      $url = $this->generateUrl($book->get('code')->value, (int) $chapter);
      // Generate the popup content and wrap the reference in a link.
      return '<a href="' . $url . '" class="bible-reference" data-reference="' . $reference . '"><span class="reference">' . $reference . '</span><span class="popup">' . $popup_content . '</span></a>';
    }
    else {
      // User doesn't have permission, return a span without link.
      return '<span class="bible-reference" data-reference="' . $reference . '"><span class="reference">' . $reference . '</span><span class="popup">' . $popup_content . '</span></span>';
    }
  }

  /**
   * Generate URL for Bible reference.
   *
   * @param string $bookCode
   *   The book code.
   * @param int $chapter
   *   The chapter number.
   *
   * @return string
   *   The generated URL.
   */
  protected function generateUrl($bookCode, $chapter) {
    // Get the Bible shortname from the book entity.
    $bookStorage = $this->entityTypeManager->getStorage('bible_book');
    $bookQuery = $bookStorage->getQuery()
      ->condition('code', $bookCode)
      ->accessCheck(FALSE)
      ->range(0, 1);

    $bookIds = $bookQuery->execute();
    if (!$bookIds) {
      return '#';
    }

    $book = $bookStorage->load(reset($bookIds));
    if (!$book) {
      return '#';
    }

    /** @var \Drupal\bible\Entity\BibleBook $book */
    /** @var \Drupal\bible\Entity\Bible|null $bible */
    $bible = $book->get('bible')->entity;
    if (!$bible) {
      return '#';
    }

    /** @var \Drupal\bible\Entity\Bible $bible */
    return Url::fromRoute('bible.read', [
      'bible' => $bible->get('shortname')->value,
      'book' => $bookCode,
      'chapter' => $chapter,
    ])->toString();
  }

}
