<?php

namespace Drupal\bible\Service;

use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Provides batch operations for Bible import.
 *
 * Contains static methods that serve as batch operation callbacks
 * for importing Bible data in chunks.
 */
class BibleBatchOperations {

  use StringTranslationTrait;

  /**
   * Batch operation: Creates a Bible entity.
   *
   * @param array $bible_data
   *   The Bible data array.
   * @param array $context
   *   The batch context array.
   */
  public static function createBible(array $bible_data, array &$context): void {
    $entity_type_manager = \Drupal::entityTypeManager();

    if (!isset($context['results']['bible_id'])) {
      $context['results']['bible_id'] = NULL;
      $context['results']['created'] = [
        'bible' => 0,
        'books' => 0,
        'verses' => 0,
      ];
      $context['results']['errors'] = [];
    }

    try {
      // Check if Bible already exists.
      $existing_bibles = $entity_type_manager
        ->getStorage('bible')
        ->loadByProperties([
          'shortname' => $bible_data['shortname'],
          'langcode' => $bible_data['langcode'],
        ]);

      if (!empty($existing_bibles)) {
        $bible = reset($existing_bibles);
        $context['results']['bible_id'] = $bible->id();
        $context['message'] = t('Using existing Bible: @name', ['@name' => $bible_data['name']]);
      }
      else {
        $bible = $entity_type_manager
          ->getStorage('bible')
          ->create([
            'shortname' => $bible_data['shortname'],
            'name' => $bible_data['name'],
            'langcode' => $bible_data['langcode'],
            'version' => $bible_data['version'] ?? NULL,
          ]);

        $bible->save();
        $context['results']['bible_id'] = $bible->id();
        $context['results']['created']['bible']++;
        $context['message'] = t('Created Bible: @name', ['@name' => $bible_data['name']]);
      }
    }
    catch (\Exception $e) {
      $context['results']['errors'][] = t('Failed to create Bible @name: @error', [
        '@name' => $bible_data['name'],
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Batch operation: Creates all book entities for a Bible using bulk insert.
   *
   * @param array $books_data
   *   Array of book data.
   * @param array $context
   *   The batch context array.
   */
  public static function createBooks(array $books_data, array &$context): void {
    if (!isset($context['results']['bible_id']) || empty($context['results']['bible_id'])) {
      $context['results']['errors'][] = t('Cannot create books: No Bible ID available');
      return;
    }

    $bible_id = $context['results']['bible_id'];
    $database = \Drupal::database();
    $created_count = 0;
    $context['results']['book_entities'] = [];

    // Start transaction for atomic operation.
    $transaction = $database->startTransaction();

    try {
      // First, load any existing books for this Bible.
      $existing_books = $database->select('bible_book', 'bb')
        ->fields('bb', ['id', 'code'])
        ->condition('bible', $bible_id)
        ->execute()
        ->fetchAllKeyed(1, 0);

      // Map existing books.
      foreach ($existing_books as $code => $id) {
        if (isset($books_data[$code])) {
          $context['results']['book_entities'][$code] = $id;
        }
      }

      // Prepare bulk insert for new books.
      $insert = $database->insert('bible_book')
        ->fields([
          'uuid',
          'bible',
          'code',
          'name',
          'shortname',
          'number',
          'chapters',
        ]);

      $has_values = FALSE;
      foreach ($books_data as $book_code => $book_data) {
        // Skip if book already exists.
        if (isset($existing_books[$book_code])) {
          continue;
        }

        // Generate UUID for the book.
        $uuid = \Drupal::service('uuid')->generate();

        $insert->values([
          'uuid' => $uuid,
          'bible' => $bible_id,
          'code' => $book_data['code'],
          'name' => $book_data['name'],
          'shortname' => $book_data['shortname'],
          'number' => $book_data['number'],
          'chapters' => $book_data['chapters'],
        ]);

        $has_values = TRUE;
        $created_count++;
      }

      // Execute bulk insert if we have new books.
      if ($has_values) {
        $insert->execute();

        // Reload all books to get the new IDs.
        $all_books = $database->select('bible_book', 'bb')
          ->fields('bb', ['id', 'code'])
          ->condition('bible', $bible_id)
          ->execute()
          ->fetchAllKeyed(1, 0);

        foreach ($all_books as $code => $id) {
          if (isset($books_data[$code])) {
            $context['results']['book_entities'][$code] = $id;
          }
        }
      }

      // Commit transaction.
      // Transaction auto-commits when out of scope if not rolled back.
    }
    catch (\Exception $e) {
      // Rollback on error.
      $transaction->rollBack();
      $context['results']['errors'][] = t('Failed to create books: @error', [
        '@error' => $e->getMessage(),
      ]);
      return;
    }

    $context['results']['created']['books'] += $created_count;
    $context['message'] = t('Processed @count books (@created created)', [
      '@count' => count($books_data),
      '@created' => $created_count,
    ]);
  }

  /**
   * Batch operation: Creates a chunk of verse entities using bulk insert.
   *
   * @param array $verses_data
   *   Array of verse data.
   * @param array $context
   *   The batch context array.
   */
  public static function createVerses(array $verses_data, array &$context): void {
    if (!isset($context['results']['bible_id']) || empty($context['results']['bible_id'])) {
      $context['results']['errors'][] = t('Cannot create verses: No Bible ID available');
      return;
    }

    if (!isset($context['results']['book_entities'])) {
      $context['results']['errors'][] = t('Cannot create verses: No book entities available');
      return;
    }

    $bible_id = $context['results']['bible_id'];
    $database = \Drupal::database();
    $created_count = 0;
    $skipped_count = 0;

    // Start transaction for atomic operation.
    $transaction = $database->startTransaction();

    try {
      // Prepare bulk insert query.
      $insert = $database->insert('bible_verse')
        ->fields([
          'uuid',
          'bible',
          'book',
          'chapter',
          'verse',
          'linemark',
          'text__value',
          'text__format',
        ]);

      // Build values for bulk insert.
      $has_values = FALSE;
      foreach ($verses_data as $verse_data) {
        $book_code = $verse_data['book_code'];

        if (!isset($context['results']['book_entities'][$book_code])) {
          $context['results']['errors'][] = t('Book @code not found for verse @chapter:@verse', [
            '@code' => $book_code,
            '@chapter' => $verse_data['chapter'],
            '@verse' => $verse_data['verse'],
          ]);
          continue;
        }

        $book_id = $context['results']['book_entities'][$book_code];

        // Generate UUID for the verse.
        $uuid = \Drupal::service('uuid')->generate();

        $insert->values([
          'uuid' => $uuid,
          'bible' => $bible_id,
          'book' => $book_id,
          'chapter' => $verse_data['chapter'],
          'verse' => $verse_data['verse'],
          'linemark' => $verse_data['linemark'] ?? '',
          'text__value' => $verse_data['text'],
          'text__format' => NULL,
        ]);

        $has_values = TRUE;
        $created_count++;
      }

      // Execute bulk insert if we have values.
      if ($has_values) {
        try {
          $insert->execute();
        }
        catch (\Exception $e) {
          // Check for duplicate key violations.
          if (strpos($e->getMessage(), 'Duplicate entry') !== FALSE) {
            // Rollback to individual inserts to find duplicates.
            $transaction->rollBack();
            $created_count = 0;

            // Re-process with individual inserts to skip duplicates.
            foreach ($verses_data as $verse_data) {
              $book_code = $verse_data['book_code'];
              if (!isset($context['results']['book_entities'][$book_code])) {
                continue;
              }

              $book_id = $context['results']['book_entities'][$book_code];
              $uuid = \Drupal::service('uuid')->generate();

              try {
                $database->insert('bible_verse')
                  ->fields([
                    'uuid' => $uuid,
                    'bible' => $bible_id,
                    'book' => $book_id,
                    'chapter' => $verse_data['chapter'],
                    'verse' => $verse_data['verse'],
                    'linemark' => $verse_data['linemark'] ?? '',
                    'text__value' => $verse_data['text'],
                    'text__format' => NULL,
                  ])
                  ->execute();
                $created_count++;
              }
              catch (\Exception $e) {
                if (strpos($e->getMessage(), 'Duplicate entry') !== FALSE) {
                  $skipped_count++;
                }
                else {
                  throw $e;
                }
              }
            }
          }
          else {
            throw $e;
          }
        }
      }

      // Commit transaction.
      // Transaction auto-commits when out of scope if not rolled back.
    }
    catch (\Exception $e) {
      // Rollback on any other error.
      $transaction->rollBack();
      $context['results']['errors'][] = t('Failed to create verses batch: @error', [
        '@error' => $e->getMessage(),
      ]);
      return;
    }

    $context['results']['created']['verses'] += $created_count;
    $message_parts = [t('Created @count verses', ['@count' => $created_count])];
    if ($skipped_count > 0) {
      $message_parts[] = t('skipped @count duplicates', ['@count' => $skipped_count]);
    }
    $context['message'] = implode(', ', $message_parts);
  }

  /**
   * Batch 'finished' callback.
   *
   * @param bool $success
   *   Whether the batch completed successfully.
   * @param array $results
   *   The batch results.
   * @param array $operations
   *   The operations that were processed.
   */
  public static function finished(bool $success, array $results, array $operations): void {
    $messenger = \Drupal::messenger();

    if ($success) {
      $message = \Drupal::translation()->formatPlural(
        $results['created']['bible'] + $results['created']['books'] + $results['created']['verses'],
        'Bible import completed successfully. Created 1 entity.',
        'Bible import completed successfully. Created @count entities.'
      );

      $details = t('Details: @bible Bibles, @books books, @verses verses', [
        '@bible' => $results['created']['bible'],
        '@books' => $results['created']['books'],
        '@verses' => $results['created']['verses'],
      ]);

      $messenger->addMessage($message);
      $messenger->addMessage($details);
    }
    else {
      $messenger->addError(t('Bible import failed with errors.'));
    }

    // Report any errors that occurred.
    if (!empty($results['errors'])) {
      foreach ($results['errors'] as $error) {
        $messenger->addError($error);
      }
    }
  }

}
