<?php

namespace Drupal\bible\Service;

use Drupal\bible\BibleParserInterface;
use Drupal\bible\Exception\BibleParseException;

/**
 * Service for parsing Bible Context files.
 *
 * Handles parsing of .bc and .bc.txt files in Bible Context format
 * and prepares data for batch import into Drupal entities.
 */
class BibleParser implements BibleParserInterface {

  /**
   * {@inheritdoc}
   */
  public function parseFile(string $uri): array {
    $fd = @fopen($uri, 'rb');
    if (empty($fd)) {
      throw new BibleParseException("Could not open file for reading: {$uri}");
    }

    $result = [
      'bible' => NULL,
      'books' => [],
      'verses' => [],
      'stats' => [
        'line_count' => 0,
        'bible_count' => 0,
        'book_count' => 0,
        'verse_count' => 0,
        'errors' => [],
      ],
    ];

    $status = '';
    $line_number = 0;
    $current_book_number = 1;

    try {
      while (!feof($fd)) {
        $line = fgets($fd, 10 * 1024);
        $line_number++;
        $result['stats']['line_count'] = $line_number;

        $line = trim(strtr($line, ["\\\n" => '']));
        if (strlen($line) > 0) {
          try {
            $this->processLine($line, $status, $result, $current_book_number, $line_number);
          }
          catch (BibleParseException $e) {
            $result['stats']['errors'][] = [
              'line' => $line_number,
              'content' => $line,
              'error' => $e->getMessage(),
            ];
            // Continue processing other lines.
          }
        }
      }
    }
    finally {
      fclose($fd);
    }

    // Validate that we have required data.
    if (empty($result['bible'])) {
      throw new BibleParseException('No Bible metadata found in file');
    }

    if (empty($result['books'])) {
      throw new BibleParseException('No book data found in file');
    }

    if (empty($result['verses'])) {
      throw new BibleParseException('No verse data found in file');
    }

    return $result;
  }

  /**
   * Processes a single line from the Bible Context file.
   *
   * @param string $line
   *   The line to process.
   * @param string $status
   *   The current parsing status (B, C, V).
   * @param array $result
   *   The result array being built.
   * @param int $current_book_number
   *   The current book number counter.
   * @param int $line_number
   *   The current line number for error reporting.
   *
   * @throws \Drupal\bible\Exception\BibleParseException
   *   Thrown when line parsing fails.
   */
  protected function processLine(string $line, string &$status, array &$result, int &$current_book_number, int $line_number): void {
    switch (substr($line, 0, 1)) {
      // Section Declare.
      case '*':
        if (strncmp('*Bible', $line, 6) == 0) {
          $status = 'B';
        }
        elseif (strncmp('*Chapter', $line, 8) == 0) {
          $status = 'C';
        }
        elseif (strncmp('*Context', $line, 8) == 0) {
          $status = 'V';
        }
        break;

      // Comment.
      case '#':
        // Skip comments.
        break;

      // Variable Setting.
      case '^':
        if (strncmp('^Bible', $line, 6) == 0) {
          $line = $this->convertCharacterEncoding($line);
          $data = explode('|', $line);
          if (count($data) < 4) {
            throw new BibleParseException('Invalid Bible declaration format', $line_number, $line);
          }
          // Skip ^Bible declarations - for updating existing Bibles.
        }
        elseif (strncmp('^Context', $line, 8) == 0) {
          $status = 'V';
        }
        break;

      // Context Data.
      default:
        $line = $this->convertCharacterEncoding($line);
        $data = explode('|', $line);

        switch ($status) {
          case 'B':
            if (count($data) < 3) {
              throw new BibleParseException('Invalid Bible data format', $line_number, $line);
            }
            $result['bible'] = [
              'shortname' => $data[0],
              'name' => $data[1],
              'langcode' => $data[2],
              'version' => $data[3] ?? NULL,
            ];
            $result['stats']['bible_count']++;
            break;

          case 'C':
            if (count($data) < 4) {
              throw new BibleParseException('Invalid book data format', $line_number, $line);
            }
            $book_code = $data[0];
            $result['books'][$book_code] = [
              'number' => $current_book_number++,
              'code' => $book_code,
              'name' => $data[1],
              'shortname' => $data[2],
              'chapters' => (int) $data[3],
            ];
            $result['stats']['book_count']++;
            break;

          case 'V':
            if (count($data) < 5) {
              throw new BibleParseException('Invalid verse data format', $line_number, $line);
            }
            $book_code = $data[0];
            $chapter = (int) $data[1];
            $verse = (int) $data[2];

            if (!isset($result['books'][$book_code])) {
              throw new BibleParseException("Verse references unknown book: {$book_code}", $line_number, $line);
            }

            if (!isset($result['verses'][$book_code])) {
              $result['verses'][$book_code] = [];
            }
            if (!isset($result['verses'][$book_code][$chapter])) {
              $result['verses'][$book_code][$chapter] = [];
            }

            $result['verses'][$book_code][$chapter][$verse] = [
              'book_code' => $book_code,
              'chapter' => $chapter,
              'verse' => $verse,
              'linemark' => $data[3],
              'text' => $data[4],
            ];
            $result['stats']['verse_count']++;
            break;
        }
        break;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateFile(string $uri): bool {
    try {
      $fd = @fopen($uri, 'rb');
      if (empty($fd)) {
        return FALSE;
      }

      $has_bible_declaration = FALSE;
      $has_book_data = FALSE;
      $has_verse_data = FALSE;
      $line_count = 0;

      while (!feof($fd) && $line_count < 100) {
        $line = fgets($fd, 10 * 1024);
        $line_count++;
        $line = trim($line);

        if (empty($line)) {
          continue;
        }

        // Check for required sections.
        if (strpos($line, '*Bible') === 0) {
          $has_bible_declaration = TRUE;
        }
        elseif (strpos($line, '*Chapter') === 0) {
          $has_book_data = TRUE;
        }
        elseif (strpos($line, '*Context') === 0) {
          $has_verse_data = TRUE;
        }

        // If we have all required sections, file is valid.
        if ($has_bible_declaration && $has_book_data && $has_verse_data) {
          fclose($fd);
          return TRUE;
        }
      }

      fclose($fd);
      return $has_bible_declaration && $has_book_data && $has_verse_data;
    }
    catch (\Exception $e) {
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedExtensions(): array {
    return ['bc', 'txt'];
  }

  /**
   * {@inheritdoc}
   */
  public function convertCharacterEncoding(string $text): string {
    $character_encoding = mb_detect_encoding($text, 'UTF-8, UTF-16, ISO-8859-1, ISO-8859-15, Windows-1252, ASCII');
    switch ($character_encoding) {
      case 'UTF-8':
        // Do nothing.
        break;

      case 'ISO-8859-1':
        $text = mb_convert_encoding($text, 'UTF-8', 'ISO-8859-1');
        break;

      default:
        $text = mb_convert_encoding($text, 'UTF-8', $character_encoding);
        break;
    }
    return $text;
  }

  /**
   * Counts total items that will be processed for batch operation sizing.
   *
   * @param array $parsed_data
   *   The parsed Bible data.
   *
   * @return int
   *   Total number of items to process.
   */
  public function countTotalItems(array $parsed_data): int {
    $total = 0;

    // 1 Bible entity
    if (!empty($parsed_data['bible'])) {
      $total++;
    }

    // Book entities.
    $total += count($parsed_data['books']);

    // Verse entities.
    foreach ($parsed_data['verses'] as $book_verses) {
      foreach ($book_verses as $chapter_verses) {
        $total += count($chapter_verses);
      }
    }

    return $total;
  }

  /**
   * Prepares data for batch processing by chunking verses.
   *
   * @param array $parsed_data
   *   The parsed Bible data.
   * @param int $chunk_size
   *   Number of verses per batch operation.
   *
   * @return array
   *   Array of batch operations.
   */
  public function prepareBatchOperations(array $parsed_data, int $chunk_size = 100): array {
    $operations = [];

    // First operation: Create Bible entity.
    $operations[] = [
      'Drupal\bible\Service\BibleBatchOperations::createBible',
      [$parsed_data['bible']],
    ];

    // Second operation: Create all book entities.
    $operations[] = [
      'Drupal\bible\Service\BibleBatchOperations::createBooks',
      [$parsed_data['books']],
    ];

    // Chunk verses into manageable batches.
    $verse_chunks = [];
    $current_chunk = [];
    $chunk_count = 0;

    foreach ($parsed_data['verses'] as $book_verses) {
      foreach ($book_verses as $chapter_verses) {
        foreach ($chapter_verses as $verse_data) {
          $current_chunk[] = $verse_data;
          $chunk_count++;

          if ($chunk_count >= $chunk_size) {
            $verse_chunks[] = $current_chunk;
            $current_chunk = [];
            $chunk_count = 0;
          }
        }
      }
    }

    // Add remaining verses.
    if (!empty($current_chunk)) {
      $verse_chunks[] = $current_chunk;
    }

    // Create batch operations for verse chunks.
    foreach ($verse_chunks as $chunk) {
      $operations[] = [
        'Drupal\bible\Service\BibleBatchOperations::createVerses',
        [$chunk],
      ];
    }

    return $operations;
  }

}
