<?php

namespace Drupal\Tests\bible\Kernel;

use Drupal\user\Entity\User;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\AnonymousUserSession;
use Drupal\user\Entity\Role;

/**
 * Tests access permissions for Bible content.
 *
 * @group bible
 */
class BibleAccessKernelTest extends BibleImportKernelTest {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'filter',
    'file',
    'views',
    'node',
    'bible',
  ];

  /**
   * The Bible parser service.
   *
   * @var \Drupal\bible\BibleParserInterface
   */
  protected $bibleParser;

  /**
   * An admin user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * An anonymous user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $anonymousUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // We need to have users and roles.
    $this->installEntitySchema('user');
    $this->installEntitySchema('user_role');

    $this->bibleParser = $this->container->get('bible.parser');

    // Create users for testing.
    $this->adminUser = User::create([
      'name' => 'admin',
      'mail' => 'admin@example.com',
      'roles' => ['administrator'],
    ]);
    $this->adminUser->save();

    $this->testImportBibleFromFile();
  }

  /**
   * Switch to given user.
   */
  protected function setCurrentUser(AccountInterface $user) {
    $this->container->get('current_user')->setAccount($user);
  }

  /**
   * Admin users can access Bible content.
   */
  public function testAdminCanAccessBible() {
    // Switch to admin user.
    $this->setCurrentUser($this->adminUser);

    $request = Request::create('/bible/KJV/GEN/1');
    $this->container->get('request_stack')->push($request);

    // // 3. Let Drupal handle it.
    $response = $this->container->get('http_kernel')->handle($request);

    $this->assertSame(200, $response->getStatusCode());
  }

  /**
   * Aanonymous users cannot access Bible content.
   */
  public function testAnonymousCannotAccessBible() {
    // Switch to anonymous user.
    $this->setCurrentUser(new AnonymousUserSession());

    $request = Request::create('/bible/KJV/GEN/1');
    $this->container->get('request_stack')->push($request);

    // // 3. Let Drupal handle it.
    $response = $this->container->get('http_kernel')->handle($request);

    $this->assertSame(403, $response->getStatusCode());
  }

  /**
   * Having "access_content" permission is not enough to read the Bible.
   */
  public function testAnonymousCannotAccessBibleWithoutSpecialPermission() {
    // Create and configure anonymous role.
    $anonymous_role = Role::create([
      'id' => 'anonymous',
      'label' => 'Anonymous user',
    ]);
    $anonymous_role->grantPermission('access content');
    $anonymous_role->save();

    // Switch to anonymous user.
    $this->setCurrentUser(new AnonymousUserSession());

    $request = Request::create('/bible/KJV/GEN/1');
    $this->container->get('request_stack')->push($request);

    // // 3. Let Drupal handle it.
    $response = $this->container->get('http_kernel')->handle($request);

    $this->assertSame(403, $response->getStatusCode());
  }

  /**
   * Having "view bible entity" permission is required.
   */
  public function testAnonymousCanAccessBibleWithViewBiblePermission() {
    // Create and configure anonymous role.
    $anonymous_role = Role::create([
      'id' => 'anonymous',
      'label' => 'Anonymous user',
    ]);
    $anonymous_role->grantPermission('access content');
    $anonymous_role->grantPermission('view bible entity');
    $anonymous_role->save();

    // Switch to anonymous user.
    $this->setCurrentUser(new AnonymousUserSession());

    $request = Request::create('/bible/KJV/GEN/1');
    $this->container->get('request_stack')->push($request);

    // // 3. Let Drupal handle it.
    $response = $this->container->get('http_kernel')->handle($request);

    $this->assertSame(200, $response->getStatusCode());
  }

}
