<?php

namespace Drupal\Tests\bible\Kernel;

use Drupal\bible\Service\BibleBatchOperations;
use Drupal\KernelTests\KernelTestBase;
use Drupal\bible\Entity\Bible;

/**
 * Tests the Bible import functionality using kernel tests.
 *
 * @group bible
 */
class BibleImportKernelTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'filter',
    'file',
    'views',
    'node',
    'bible',
  ];

  /**
   * The Bible parser service.
   *
   * @var \Drupal\bible\BibleParserInterface
   */
  protected $bibleParser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Ask Drupal to enable bible and everything it depends on.
    \Drupal::service('module_installer')
      ->install(['bible'], TRUE);

    // Install entitites so required tables exists.
    $this->installEntitySchema('bible');
    $this->installEntitySchema('bible_book');
    $this->installEntitySchema('bible_verse');
    // $this->installEntitySchema('bible_note');
    // $this->installEntitySchema('bible_concordance');
    $this->bibleParser = $this->container->get('bible.parser');
  }

  /**
   * Tests importing a Bible from a mock file.
   */
  public function testImportBibleFromFile() {
    // Create a mock Bible Context file content.
    $bc_content = <<<EOT
^Bible|KJV|King James Version|en|1611
*Bible
KJV|King James Version|en|1611
*Chapter
GEN|Genesis|Gen|50
EXO|Exodus|Exo|40
LEV|Leviticus|Lev|27
*Context
GEN|1|1||In the beginning God created the heaven and the earth.
GEN|1|2||And the earth was without form, and void; and darkness was upon the face of the deep. And the Spirit of God moved upon the face of the waters.
GEN|1|3||And God said, Let there be light: and there was light.
EXO|1|1||Now these are the names of the children of Israel, which came into Egypt; every man and his household came with Jacob.
EOT;

    // Save the content to a temporary file.
    $filename = 'test_kjv_bible.bc.txt';
    $uri = 'public://' . $filename;
    file_put_contents($uri, $bc_content);

    // Validate the file.
    $is_valid = $this->bibleParser->validateFile($uri);
    $this->assertTrue($is_valid, 'Bible file should be valid');

    // Parse the file.
    $parsed_data = $this->bibleParser->parseFile($uri);

    // Now test the batch operations.
    $batch_context = [];

    // Create Bible entity.
    BibleBatchOperations::createBible(
      $parsed_data['bible'],
      $batch_context
    );

    // Make sure there are no errors.
    $this->assertEmpty($batch_context['results']['errors']);

    $this->assertArrayHasKey('results', $batch_context);
    $this->assertArrayHasKey('bible_id', $batch_context['results']);

    $bible_id = $batch_context['results']['bible_id'];
    $this->assertNotNull($bible_id);
    $bible = Bible::load($bible_id);
    $this->assertNotNull($bible);
    $this->assertEquals('KJV', $bible->shortname->value);
    $this->assertEquals('King James Version', $bible->name->value);

    // Create books.
    BibleBatchOperations::createBooks(
      $parsed_data['books'],
      $batch_context
    );

    $books = \Drupal::entityTypeManager()
      ->getStorage('bible_book')
      ->loadByProperties(['bible' => $bible_id]);
    $this->assertCount(3, $books);

    // Find Genesis book ID.
    $genesis_id = NULL;
    foreach ($books as $book) {
      if ($book->code->value === 'GEN') {
        $genesis_id = $book->id();
        break;
      }
    }
    $this->assertNotNull($genesis_id);

    // Create verses.
    $book_map = [];
    foreach ($books as $book) {
      $book_map[$book->code->value] = $book->id();
    }

    // Flatten verses array for batch operations.
    $flat_verses = [];
    foreach ($parsed_data['verses'] as $chapters) {
      foreach ($chapters as $verses) {
        foreach ($verses as $verse_data) {
          // verse_data already has book_code field.
          $flat_verses[] = $verse_data;
        }
      }
    }

    BibleBatchOperations::createVerses(
      $flat_verses,
      $batch_context
    );

    // Verify verses were created.
    $verses = \Drupal::entityTypeManager()
      ->getStorage('bible_verse')
      ->loadByProperties([
        'bible' => $bible_id,
        'book' => $genesis_id,
      ]);
    $this->assertCount(3, $verses);

    // Check Genesis 1:1.
    $gen_1_1 = \Drupal::entityTypeManager()
      ->getStorage('bible_verse')
      ->loadByProperties([
        'bible' => $bible_id,
        'book' => $genesis_id,
        'chapter' => 1,
        'verse' => 1,
      ]);
    $this->assertCount(1, $gen_1_1);
    $verse = reset($gen_1_1);
    $this->assertEquals('In the beginning God created the heaven and the earth.', $verse->text->value);
  }

}
