<?php

namespace Drupal\Tests\bible\Unit\Plugin\Filter;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Database\Connection;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the BibleFilter plugin functionality.
 *
 * @group bible
 * @coversDefaultClass \Drupal\bible\Plugin\Filter\BibleFilter
 */
class BibleFilterTest extends UnitTestCase {

  /**
   * The BibleFilter plugin.
   *
   * @var \Drupal\bible\Plugin\Filter\BibleFilter
   */
  protected $filter;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Mock book storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $bookStorage;

  /**
   * Mock verse storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $verseStorage;

  /**
   * Mock database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->bookStorage = $this->createMock(EntityStorageInterface::class);
    $this->verseStorage = $this->createMock(EntityStorageInterface::class);
    $this->database = $this->createMock(Connection::class);

    // Mock query objects.
    $bookQuery = $this->createMock('\Drupal\Core\Entity\Query\QueryInterface');
    $verseQuery = $this->createMock('\Drupal\Core\Entity\Query\QueryInterface');

    // Mock the book storage to return the book query.
    $this->bookStorage->method('getQuery')
      ->willReturn($bookQuery);

    // Mock the verse storage to return the verse query.
    $this->verseStorage->method('getQuery')
      ->willReturn($verseQuery);

    // Mock the entity type manager.
    $this->entityTypeManager->method('getStorage')
      ->willReturnMap([
        ['bible_book', $this->bookStorage],
        ['bible_verse', $this->verseStorage],
      ]);

    // Configure the book query to return appropriate book based on query.
    $bookQuery->method('accessCheck')
      ->willReturnSelf();
    $bookQuery->method('range')
      ->willReturnSelf();

    $orGroup = $this->createMock('\Drupal\Core\Entity\Query\ConditionInterface');
    $bookQuery->method('orConditionGroup')
      ->willReturn($orGroup);
    $orGroup->method('condition')
      ->willReturnSelf();
    $bookQuery->method('condition')
      ->willReturnSelf();
    $bookQuery->method('execute')
      ->willReturn([123]);

    // Mock a book entity.
    $book = $this->createMock('\Drupal\bible\Entity\BibleBook');
    $book->method('get')->willReturnMap([
      ['code', (object) ['value' => 'JHN']],
      ['id', 123],
    ]);
    $book->method('id')->willReturn(123);
    $this->bookStorage->method('loadMultiple')
      ->willReturn([$book]);

    // Configure the verse query.
    $verseQuery->method('accessCheck')
      ->willReturnSelf();
    $verseQuery->method('condition')
      ->willReturnSelf();
    $verseQuery->method('execute')
      ->willReturn([456]);

    // Mock a verse entity.
    $verse = $this->createMock('\Drupal\bible\Entity\BibleVerse');
    $verse->method('get')->willReturnMap([
      ['text', (object) ['value' => 'For God so loved the world...']],
    ]);
    $this->verseStorage->method('loadMultiple')
      ->with([456])
      ->willReturn([$verse]);

    // Create a test-specific filter that avoids URL generation.
    $this->filter = new TestableBibleFilter(
      [],
      'bible_filter',
      ['provider' => 'bible'],
      $this->entityTypeManager,
      $this->database
    );
  }

  /**
   * Tests basic pattern matching for Bible references.
   *
   * @covers ::doProcess
   */
  public function testDoProcessBasicMatching() {
    $input_text = 'Check out John 3:16 for inspiration.';
    $result = $this->filter->doProcess($input_text, ["John", "Genesis"]);

    $processed_text = $result->getProcessedText();
    $this->assertStringContainsString('class="bible-reference"', $processed_text);
    $this->assertStringContainsString('John 3:16', $processed_text);
  }

  /**
   * Tests Psalm reference without trailing dot.
   *
   * @covers ::doProcess
   */
  public function testPsalmReferenceWithoutTrailingDot() {
    $input_text = '<p>“that it may be displayed because of the truth.” Psalm 60:4</p>';
    $result = $this->filter->doProcess($input_text, [
      "Genesis", "GEN", "Exodus", "EXO", "Psalms", "PS",
      "Psalm", "Jude", "JUD", "Revelation", "REV",
    ]);

    $processed_text = $result->getProcessedText();
    $this->assertStringContainsString('class="bible-reference"', $processed_text);
    $this->assertStringContainsString('Psalm 60:4', $processed_text);
  }

}
