<?php

namespace Drupal\Tests\bible\Unit\Service;

use Drupal\bible\Exception\BibleParseException;
use Drupal\bible\Service\BibleParser;
use Drupal\Tests\UnitTestCase;
use org\bovigo\vfs\vfsStream;

/**
 * Tests for the BibleParser service.
 *
 * @coversDefaultClass \Drupal\bible\Service\BibleParser
 * @group bible
 */
class BibleParserTest extends UnitTestCase {

  /**
   * The BibleParser service under test.
   *
   * @var \Drupal\bible\Service\BibleParser
   */
  protected $bibleParser;

  /**
   * Virtual file system for testing file operations.
   *
   * @var \org\bovigo\vfs\vfsStreamDirectory
   */
  protected $vfsRoot;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->bibleParser = new BibleParser();
    $this->vfsRoot = vfsStream::setup('test');
  }

  /**
   * Tests parsing a valid Bible Context file.
   *
   * @covers ::parseFile
   */
  public function testParseValidBibleFile() {
    $content = $this->getValidBibleContent();
    $file = vfsStream::newFile('test.bc')->at($this->vfsRoot)->setContent($content);

    $result = $this->bibleParser->parseFile($file->url());

    // Test Bible data.
    $this->assertArrayHasKey('bible', $result);
    $this->assertEquals('KJV', $result['bible']['shortname']);
    $this->assertEquals('King James Version', $result['bible']['name']);
    $this->assertEquals('en', $result['bible']['langcode']);
    $this->assertEquals('1611', $result['bible']['version']);

    // Test book data.
    $this->assertArrayHasKey('books', $result);
    $this->assertArrayHasKey('GEN', $result['books']);
    $this->assertEquals('Genesis', $result['books']['GEN']['name']);
    $this->assertEquals('Gen', $result['books']['GEN']['shortname']);
    $this->assertEquals(50, $result['books']['GEN']['chapters']);

    // Test verse data.
    $this->assertArrayHasKey('verses', $result);
    $this->assertArrayHasKey('GEN', $result['verses']);
    $this->assertArrayHasKey(1, $result['verses']['GEN']);
    $this->assertArrayHasKey(1, $result['verses']['GEN'][1]);
    $this->assertEquals('In the beginning God created the heaven and the earth.',
      $result['verses']['GEN'][1][1]['text']);

    // Test statistics.
    $this->assertArrayHasKey('stats', $result);
    $this->assertEquals(1, $result['stats']['bible_count']);
    $this->assertEquals(1, $result['stats']['book_count']);
    $this->assertEquals(2, $result['stats']['verse_count']);
    $this->assertEmpty($result['stats']['errors']);
  }

  /**
   * Tests parsing an invalid Bible Context file.
   *
   * @covers ::parseFile
   */
  public function testParseInvalidBibleFile() {
    $content = "Invalid content\nNo proper structure";
    $file = vfsStream::newFile('invalid.bc')->at($this->vfsRoot)->setContent($content);

    $this->expectException(BibleParseException::class);
    $this->expectExceptionMessage('No Bible metadata found in file');

    $this->bibleParser->parseFile($file->url());
  }

  /**
   * Tests parsing a file that cannot be opened.
   *
   * @covers ::parseFile
   */
  public function testParseNonExistentFile() {
    $this->expectException(BibleParseException::class);
    $this->expectExceptionMessage('Could not open file for reading');

    $this->bibleParser->parseFile('non-existent-file.bc');
  }

  /**
   * Tests parsing a file with malformed Bible data.
   *
   * @covers ::parseFile
   */
  public function testParseMalformedBibleData() {
    // Include minimal valid structure but with malformed verse data.
    $content = implode("\n", [
      '*Bible',
      'KJV|King James Version|en|1611',
      '*Chapter',
    // Valid book data.
      'GEN|Genesis|Gen|50',
      '*Context',
    // Invalid: missing verse number.
      'GEN|1||Missing verse number|Text here',
    // Invalid: missing text.
      'GEN|2|1|',
    ]);
    $file = vfsStream::newFile('malformed.bc')->at($this->vfsRoot)->setContent($content);

    $result = $this->bibleParser->parseFile($file->url());

    // Should have parsing errors but continue processing.
    $this->assertNotEmpty($result['stats']['errors']);
    $this->assertGreaterThanOrEqual(1, count($result['stats']['errors']));
    // Should still have parsed the valid Bible and book data.
    $this->assertNotNull($result['bible']);
    $this->assertNotEmpty($result['books']);
  }

  /**
   * Tests file validation with a valid Bible Context file.
   *
   * @covers ::validateFile
   */
  public function testValidateValidFile() {
    $content = $this->getValidBibleContent();
    $file = vfsStream::newFile('valid.bc')->at($this->vfsRoot)->setContent($content);

    $this->assertTrue($this->bibleParser->validateFile($file->url()));
  }

  /**
   * Tests file validation with an invalid file.
   *
   * @covers ::validateFile
   */
  public function testValidateInvalidFile() {
    $content = "Just some random text\nNo Bible structure here";
    $file = vfsStream::newFile('invalid.bc')->at($this->vfsRoot)->setContent($content);

    $this->assertFalse($this->bibleParser->validateFile($file->url()));
  }

  /**
   * Tests file validation with a non-existent file.
   *
   * @covers ::validateFile
   */
  public function testValidateNonExistentFile() {
    $this->assertFalse($this->bibleParser->validateFile('non-existent.bc'));
  }

  /**
   * Tests getting supported file extensions.
   *
   * @covers ::getSupportedExtensions
   */
  public function testGetSupportedExtensions() {
    $extensions = $this->bibleParser->getSupportedExtensions();

    $this->assertIsArray($extensions);
    $this->assertContains('bc', $extensions);
    $this->assertContains('txt', $extensions);
  }

  /**
   * Tests character encoding conversion.
   *
   * @covers ::convertCharacterEncoding
   */
  public function testConvertCharacterEncoding() {
    // Test UTF-8 (should remain unchanged)
    $utf8_text = 'This is UTF-8 text';
    $result = $this->bibleParser->convertCharacterEncoding($utf8_text);
    $this->assertEquals($utf8_text, $result);

    // Test ISO-8859-1 conversion.
    $iso_text = mb_convert_encoding('Café', 'ISO-8859-1', 'UTF-8');
    $result = $this->bibleParser->convertCharacterEncoding($iso_text);
    $this->assertEquals('Café', $result);
  }

  /**
   * Tests counting total items in parsed data.
   *
   * @covers ::countTotalItems
   */
  public function testCountTotalItems() {
    $parsed_data = [
      'bible' => ['shortname' => 'KJV'],
      'books' => [
        'GEN' => ['name' => 'Genesis'],
        'EXO' => ['name' => 'Exodus'],
      ],
      'verses' => [
        'GEN' => [
          1 => [1 => ['text' => 'Verse 1'], 2 => ['text' => 'Verse 2']],
          2 => [1 => ['text' => 'Verse 3']],
        ],
        'EXO' => [
          1 => [1 => ['text' => 'Verse 4']],
        ],
      ],
    ];

    $total = $this->bibleParser->countTotalItems($parsed_data);

    // 1 Bible + 2 Books + 4 Verses = 7 total items
    $this->assertEquals(7, $total);
  }

  /**
   * Tests preparing batch operations.
   *
   * @covers ::prepareBatchOperations
   */
  public function testPrepareBatchOperations() {
    $parsed_data = [
      'bible' => ['shortname' => 'KJV'],
      'books' => [
        'GEN' => ['name' => 'Genesis'],
      ],
      'verses' => [
        'GEN' => [
          1 => [
            1 => ['text' => 'Verse 1'],
            2 => ['text' => 'Verse 2'],
            3 => ['text' => 'Verse 3'],
          ],
        ],
      ],
    ];

    $operations = $this->bibleParser->prepareBatchOperations($parsed_data, 2);

    // Should have: 1 Bible + 1 Books + 2 Verse chunk operations.
    $this->assertCount(4, $operations);

    // Check operation structure.
    $this->assertEquals('Drupal\bible\Service\BibleBatchOperations::createBible',
      $operations[0][0]);
    $this->assertEquals('Drupal\bible\Service\BibleBatchOperations::createBooks',
      $operations[1][0]);
    $this->assertEquals('Drupal\bible\Service\BibleBatchOperations::createVerses',
      $operations[2][0]);
    $this->assertEquals('Drupal\bible\Service\BibleBatchOperations::createVerses',
      $operations[3][0]);
  }

  /**
   * Tests parsing a file with Unicode BOM.
   *
   * @covers ::parseFile
   */
  public function testParseFileWithBom() {
    // Add UTF-8 BOM to content.
    $content = "\xEF\xBB\xBF" . $this->getValidBibleContent();
    $file = vfsStream::newFile('bom.bc')->at($this->vfsRoot)->setContent($content);

    $result = $this->bibleParser->parseFile($file->url());

    // Should still parse correctly despite BOM.
    $this->assertArrayHasKey('bible', $result);
    $this->assertEquals('KJV', $result['bible']['shortname']);
  }

  /**
   * Returns valid Bible Context file content for testing.
   *
   * @return string
   *   Valid Bible Context content.
   */
  protected function getValidBibleContent(): string {
    return implode("\n", [
      '# Bible Context File',
      '*Bible',
      'KJV|King James Version|en|1611',
      '*Chapter',
      'GEN|Genesis|Gen|50',
      '*Context',
      'GEN|1|1||In the beginning God created the heaven and the earth.',
      'GEN|1|2||And the earth was without form, and void.',
    ]);
  }

}
