import { igniteEngine, Message } from 'multi-llm-ts/dist/index.js';
import { ButtonView } from 'ckeditor5/src/ui.js';
import { env } from 'ckeditor5/src/utils.js';
import { PromptHelper } from './util/prompt.js';
import { ProcessContentHelper } from './util/process-content.js';
import { HtmlParser } from './util/htmlparser.js';
import { getErrorMessages } from './util/translations.js';
import { moderateContent } from './util/moderate-content.js';
import { AIApi } from './util/ai-api.js';
import { checkModel } from './util/check-model.js';
import { aiAgentContext } from './aiagentcontext.js';
import { AI_ENGINE } from './const.js';
export default class AiAgentService {
    editor;
    aiEngine;
    aiModel;
    apiKey;
    endpointUrl;
    temperature;
    timeOutDuration;
    maxTokens;
    retryAttempts;
    streamContent;
    stopSequences;
    aiAgentFeatureLockId = Symbol('ai-agent-feature');
    promptHelper;
    htmlParser;
    providers;
    isInlineInsertion = false;
    abortGeneration = false;
    moderationKey;
    moderationEnable;
    disableFlags = [];
    stream;
    writesPerSecond;
    processContentHelper;
    FILTERED_STRINGS = /```html|```|html\n|@@@cursor@@@/g;
    /**
     * Initializes the AiAgentService with the provided editor and configuration settings.
     *
     * @param editor - The CKEditor instance to be used with the AI assist service.
     */
    constructor(editor) {
        this.editor = editor;
        this.promptHelper = new PromptHelper(editor);
        this.htmlParser = new HtmlParser(editor);
        this.processContentHelper = new ProcessContentHelper(editor);
        const config = editor.config.get('aiAgent');
        this.aiModel = config.model;
        this.apiKey = config.apiKey;
        this.aiEngine = config.engine;
        this.endpointUrl = config.endpointUrl;
        this.temperature = config.temperature;
        this.timeOutDuration = config.timeOutDuration ?? 120000;
        this.maxTokens = config.maxOutputTokens ?? config.maxTokens;
        this.retryAttempts = config.retryAttempts;
        this.stopSequences = config.stopSequences;
        this.streamContent = config.streamContent ?? true;
        this.moderationKey = config.moderationKey ?? '';
        this.moderationEnable = config.moderationEnable ?? false;
        this.disableFlags = config.moderationDisableFlags ?? [];
        this.writesPerSecond = config.writesPerSecond ?? 10;
        this.providers = config.providers;
    }
    /**
     * Handles the slash command input from the user, processes it, and interacts with the AI model.
     *
     * @returns A promise that resolves when the command has been processed.
     */
    async handleSlashCommand(command) {
        const editor = this.editor;
        const model = editor.model;
        const mapper = editor.editing.mapper;
        const view = editor.editing.view;
        const root = model.document.getRoot();
        let content;
        let selectedContent;
        let parentEquivalentHTML;
        let parent;
        const position = model.document.selection.getLastPosition();
        if (position && root) {
            parent = position.parent;
            const inlineSlash = parent.name === 'inline-slash' ? parent : undefined;
            const equivalentView = mapper.toViewElement(parent);
            parentEquivalentHTML = equivalentView ? view.domConverter.mapViewToDom(equivalentView) : undefined;
            if (inlineSlash) {
                editor.model.change(writer => {
                    const endPosition = writer.createPositionAt(inlineSlash, 'end');
                    writer.setSelection(endPosition);
                });
                this.isInlineInsertion = true;
                const startPosition = editor.model.createPositionAt(inlineSlash, 0);
                const endPosition = editor.model.createPositionAt(inlineSlash, 'end');
                const range = model.createRange(startPosition, endPosition);
                parentEquivalentHTML = equivalentView?.parent ?
                    view.domConverter.mapViewToDom(equivalentView.parent) :
                    undefined;
                content = '';
                for (const item of range.getItems()) {
                    if (item.is('$textProxy')) {
                        content += item.data.trim(); // Add text data
                    }
                }
            }
            else if (parentEquivalentHTML) {
                content = parentEquivalentHTML?.innerText;
            }
        }
        if (command) {
            const selection = model.document.selection;
            const selectedContentFragment = model.getSelectedContent(selection);
            const viewFragment = editor.data.toView(selectedContentFragment);
            const html = editor.data.processor.toData(viewFragment);
            content = command;
            selectedContent = html;
        }
        if (this.moderationEnable) {
            const moderateContentData = await moderateContent({
                content: content,
                moderationKey: this.moderationKey,
                timeOutDuration: this.timeOutDuration,
                disableFlags: this.disableFlags,
                t: editor.t
            });
            if (!moderateContentData) {
                return;
            }
        }
        try {
            aiAgentContext.showLoader(editor);
            const prompt = await this.promptHelper.generateGptPromptBasedOnUserPrompt(content, parentEquivalentHTML?.innerHTML, selectedContent);
            if (parent && prompt) {
                await this.fetchAndProcessGptResponse(!!command, prompt, parent);
            }
        }
        catch (error) {
            console.error('Error handling slash command:', error);
            throw error;
        }
        finally {
            this.isInlineInsertion = false;
            aiAgentContext.hideLoader(editor);
        }
    }
    /**
     * Fetches and processes the GPT response based on the provided prompt and parent element.
     *
     * @param prompt - The prompt to send to the GPT model.
     * @param parent - The parent element in the editor where the response will be inserted.
     * @param retries - The number of retry attempts for the API call (default is the configured retry attempts).
     * @returns A promise that resolves when the response has been processed.
     */
    async fetchAndProcessGptResponse(command, prompt, parent, retries = this.retryAttempts) {
        const editor = this.editor;
        const t = editor.t;
        const controller = new AbortController();
        // Create a timeout that can be reset
        let timeoutId;
        const resetTimeout = () => {
            if (timeoutId) {
                clearTimeout(timeoutId);
            }
            timeoutId = setTimeout(() => controller.abort(), this.timeOutDuration);
        };
        // Set initial timeout
        resetTimeout();
        const blockID = `ai-${new Date().getTime()}`;
        try {
            let llm;
            let response;
            if (AI_ENGINE.includes(this.aiEngine)) {
                const config = {
                    apiKey: this.apiKey
                };
                const { success, error } = await checkModel(this.aiEngine, this.aiModel, this.apiKey);
                if (!success) {
                    aiAgentContext.showError(`${t('Invalid AI model specified. Available models')}: ${error} `);
                    return;
                }
                llm = igniteEngine(this.aiEngine, config);
                const messages = [
                    new Message('system', this.promptHelper.getSystemPrompt(this.isInlineInsertion)),
                    new Message('user', prompt)
                ];
                const completionOpts = {
                    maxTokens: this.maxTokens,
                    ...(this.temperature !== undefined && { temperature: this.temperature })
                };
                if (this.streamContent) {
                    // Streaming path
                    const stream = this.generate(llm, this.aiModel, messages, completionOpts);
                    await this.handleStreamingResponse(stream, blockID, parent, command, controller, llm, resetTimeout);
                }
                else {
                    // Non-streaming path
                    const result = await llm.complete(this.aiModel, messages, completionOpts);
                    if (!result.content) {
                        throw new Error(t('Empty response from AI model'));
                    }
                    await this.handleNonStreamingResponse(result.content, blockID, parent, command);
                }
            }
            else {
                const config = {
                    apiKey: this.apiKey,
                    baseURL: this.endpointUrl,
                    engine: this.aiEngine,
                    editor: this.editor
                };
                // Add providers if engine is dxai and providers is set
                if (this.aiEngine === 'dxai' && this.providers) {
                    config.providers = this.providers;
                }
                const llmCustom = new AIApi(config);
                const messages = {
                    system: this.promptHelper.getSystemPrompt(this.isInlineInsertion),
                    user: prompt
                };
                response = llmCustom.fetchAIStream(this.aiModel, messages, {
                    temperature: this.temperature,
                    max_tokens: this.maxTokens,
                    stop: this.stopSequences
                }, controller, retries);
                await this.handleStreamingResponse(response, blockID, parent, command, controller, llm, resetTimeout);
            }
        }
        catch (error) {
            if (this.abortGeneration) {
                return;
            }
            console.error('Error in fetchAndProcessGptResponse:', error);
            let errorMessage = t('We couldn\'t connect to the AI. Please check your internet');
            if (error?.status) {
                errorMessage = getErrorMessages(error.status, t);
            }
            else {
                errorMessage = error?.message?.trim();
            }
            aiAgentContext.showError(errorMessage);
            this.processContentHelper.processCompleted(blockID);
        }
        finally {
            if (timeoutId) {
                clearTimeout(timeoutId);
            }
            this.editor.disableReadOnlyMode(this.aiAgentFeatureLockId);
        }
    }
    async handleStreamingResponse(stream, blockID, parent, command, controller, llm, resetTimeout) {
        let isFirstChunk = true;
        let contentBuffer = '';
        const updateInterval = 1000 / this.writesPerSecond; // Calculate interval in ms
        const updateContent = async () => {
            if (contentBuffer) {
                await this.processContentHelper.updateContent(contentBuffer, blockID);
            }
        };
        const updateContentTimer = setInterval(updateContent, updateInterval);
        try {
            for await (const c of stream) {
                if (isFirstChunk) {
                    aiAgentContext.hideLoader(this.editor);
                    this.cancelGenerationButton(blockID, controller, llm);
                    this.processContentHelper.undoRedoHandler();
                    this.processContentHelper.insertAiTag(blockID);
                    this.processContentHelper.clearParentContent(parent, command);
                    isFirstChunk = false;
                }
                const chunk = c;
                if (chunk.type === 'status') {
                    await this.processContentHelper.animatedStatusMessages(chunk.text, blockID);
                }
                // Filter out markdown code blocks and normalize content
                const filteredText = chunk.text
                    .replace(this.FILTERED_STRINGS, '');
                if (chunk.type === 'content') {
                    contentBuffer += filteredText;
                }
                // Reset timeout when data is received
                resetTimeout();
            }
        }
        finally {
            clearInterval(updateContentTimer);
            await updateContent();
            this.processContentHelper.processCompleted(blockID);
            contentBuffer = '';
        }
        this.processContentHelper.processCompleted(blockID);
    }
    async handleNonStreamingResponse(content, blockID, parent, command) {
        aiAgentContext.hideLoader(this.editor);
        this.processContentHelper.insertAiTag(blockID);
        this.processContentHelper.clearParentContent(parent, command);
        // Filter out markdown code blocks and normalize content
        const filteredContent = content
            .replace(this.FILTERED_STRINGS, '');
        if (filteredContent) {
            await this.htmlParser.insertSimpleHtml(filteredContent);
        }
        this.processContentHelper.processCompleted(blockID);
    }
    /**
     * Creates and configures a cancel generation button with keyboard shortcut support.
     *
     * @param blockID - Unique identifier for the AI generation block
     * @param controller - AbortController to cancel the ongoing AI generation
     * @private
     */
    cancelGenerationButton(blockID, controller, llm) {
        const editor = this.editor;
        const t = editor.t;
        const view = new ButtonView();
        let label = t('Cancel Generation');
        if (env.isMac) {
            label = `\u2318 + \u232B ${t('Cancel Generation')}`;
        }
        if (env.isWindows) {
            label = `Ctrl + \u232B ${t('Cancel Generation')}`;
        }
        view.set({
            label,
            withText: true,
            class: 'ck-cancel-request-button'
        });
        view.on('execute', () => {
            this.abortGeneration = true;
            if (llm) {
                llm.stop(this.stream);
            }
            else {
                controller.abort();
            }
            this.processContentHelper.processCompleted(blockID);
        });
        view.render();
        editor.keystrokes.set('Ctrl+Backspace', (keyEvtData, cancel) => {
            if (keyEvtData.ctrlKey || keyEvtData.metaKey) {
                this.abortGeneration = true;
                if (llm) {
                    llm.stop(this.stream);
                }
                else {
                    controller.abort();
                }
                this.processContentHelper.processCompleted(blockID);
            }
            cancel();
        });
        const toolbarElement = editor.ui.view.toolbar.element;
        if (toolbarElement && view.element) {
            const panelContent = toolbarElement.querySelector('.ck-toolbar__items');
            if (panelContent) {
                panelContent.append(view.element);
            }
        }
        setTimeout(() => view.set({ class: 'ck-cancel-request-button visible' }), 2000);
    }
    /**
     * Generates a stream of messages from the specified language model (LLM) based on the provided input thread.
     * This method handles the streaming of responses, yielding each message as it is received.
     *
     * @param llm - The language model instance used for generating responses.
     * @param model - The identifier of the model to be used for generation.
     * @param thread - An array of messages that form the context for the generation.
     * @param opts - Options for the LLM completion, such as max tokens and temperature.
     * @returns An async generator that yields messages from the LLM as they are received.
     *
     * @throws Will throw an error if the streaming process fails or if the model is invalid.
     */
    async *generate(llm, model, thread, opts) {
        const chatModel = llm.toModel(model);
        const response = await llm.stream(chatModel, thread, opts);
        this.stream = response?.stream;
        while (true) {
            let stream2 = null;
            for await (const chunk of this.stream) {
                const stream3 = llm.nativeChunkToLlmChunk(chunk, response.context);
                for await (const msg of stream3) {
                    if (msg.type === "stream") {
                        stream2 = msg.stream;
                    }
                    else {
                        if (stream2 !== null && msg.type === "content" && msg.done) {
                            msg.done = false;
                        }
                        yield msg;
                    }
                }
            }
            if (!stream2)
                break;
            this.stream = stream2;
        }
    }
}
