<?php

namespace Drupal\entity_back_reference\Services;

use Drupal\Core\Entity\EntityFieldManager;
use Drupal\Core\Entity\EntityTypeManager;
use Drupal\field\Entity\FieldConfig;

/**
 * Identifies backreferences from entities to the specified entity.
 */
class BackReferenceFinder {

  /**
   * The entity query factory service.
   *
   * @var \Drupal\Core\Entity\Query\Sql\QueryFactory
   */
  protected $entityQuery;

  /**
   * Entity Type Manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManager
   */
  protected $entityTypeManager;

  /**
   * Entity Field Manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManager
   */
  protected $entityFieldManager;

  /**
   * Array of all fields that can reference the given entity/bundle.
   *
   * @var array
   */
  protected $backReferenceFields = [];

  /**
   * Stores an array of entity IDs that are backreferencing.
   *
   * @var array
   */
  protected $backReferenceEntityIds = [];

  /**
   * Constructs a \Drupal\entity_back_reference\BackReferenceFinder.
   *
   * @param \Drupal\Core\Entity\EntityTypeManager $entityTypeManager
   *   Entity Type Manager.
   * @param \Drupal\Core\Entity\EntityFieldManager $entityFieldManager
   *   Entity Field Manager.
   */
  public function __construct(
    EntityTypeManager $entityTypeManager,
    EntityFieldManager $entityFieldManager,
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->entityFieldManager = $entityFieldManager;
  }

  /**
   * Load all entities that reference the entity of the given identifier.
   *
   * This method does not perform any checks on the targeted entity to verify
   * the applicability of the field, as those are assumed to have been handled
   * as part of deriving a confirmed FieldConfig object.
   *
   * @param \Drupal\field\Entity\FieldConfig $field
   *   Field config.
   * @param string $targetId
   *   Target ID.
   *
   * @return \Drupal\Core\Entity\EntityInterface[]
   *   Array of Entities that reference our specified target ID.
   */
  public function loadBackReferencedEntities(FieldConfig $field, $targetId): array {
    $cid = $field->getTargetEntityTypeId() . $field->getName() . $targetId;
    $this->entityQuery = $this->entityTypeManager->getStorage(
          $field->getTargetEntityTypeId()
      )->getQuery();
    if (!isset($this->backReferenceEntityIds[$cid])) {
      $query = $this->entityQuery
        ->condition($field->getName(), $targetId)
        ->accessCheck(FALSE);
      $this->backReferenceFields[$cid] = $query->execute();
    }
    $ids = $this->backReferenceFields[$cid];

    return $this->entityTypeManager->getStorage($field->getTargetEntityTypeId())
      ->loadMultiple($ids);
  }

  /**
   * Retrieves every reference field which can point at the current entity.
   *
   * @param string $entityTypeId
   *   entity type ID.
   * @param string $entityBundleId
   *   entity bundle ID.
   *
   * @return \Drupal\field\Entity\FieldConfig[]
   *   Array of all applicable fields keyed by field name.
   */
  public function getReferencingFieldList(
    string $entityTypeId,
    string $entityBundleId,
  ): array {
    $cid = $entityTypeId . $entityBundleId;
    if (isset($this->backReferenceFields[$cid])) {
      return $this->backReferenceFields[$cid];
    }
    $entityReferenceFields = $this->entityFieldManager
      ->getFieldMapByFieldType('entity_reference');
    $fields = [];
    foreach ($entityReferenceFields as $entityTypeFieldIsOn => $fieldInfo) {
      foreach ($fieldInfo as $fieldName => $fieldData) {
        foreach ($fieldData['bundles'] as $entityBundleFieldIsOn) {
          /** @var \Drupal\field\Entity\FieldConfig */
          $field = FieldConfig::loadByName(
                $entityTypeFieldIsOn,
                $entityBundleFieldIsOn,
                $fieldName
            );

          // Check to see if the field is applicable to our entity
          // and check for references if so.
          if ($field
                && static::referenceFieldAppliesToEntity(
                    $field,
                    $entityTypeId,
                    $entityBundleId
                )
            ) {
            $fields[$fieldName] = $field;
          }
        }
      }
    }
    $this->backReferenceFields[$entityTypeId . $entityBundleId] = $fields;

    return $this->backReferenceFields[$cid];
  }

  /**
   * Identifies if supplied field is applicable to the given entity.
   *
   * @param \Drupal\field\Entity\FieldConfig $field
   *   Field configuration.
   * @param string $entityTypeId
   *   Entity type ID.
   * @param string $entityBundleId
   *   Entity bundle ID.
   *   Defaults to NULL.
   *
   * @return bool
   *   TRUE if the field is applicable, FALSE otherwise.
   */
  public static function referenceFieldAppliesToEntity(
    FieldConfig $field,
    string $entityTypeId,
    string $entityBundleId = NULL,
  ): bool {
    $entityTypeTargetedByField = $field->getSetting('target_type');
    $field_handler = $field->getSetting('handler_settings');

    return $entityTypeTargetedByField == $entityTypeId &&
        isset($field_handler['target_bundles']) &&
        isset($field_handler['target_bundles'][$entityBundleId]);
  }

}
