<?php

declare(strict_types=1);

namespace Drupal\environment_indicator\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Site\Settings;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

/**
 * Provides contextual info about the active environment indicator.
 */
class EnvironmentIndicator {
  use StringTranslationTrait;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The settings service.
   *
   * @var \Drupal\Core\Site\Settings
   */
  protected $settings;

  /**
   * The active environment indicator configuration object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $activeEnvironment;

  /**
   * The active configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $activeConfig;

  /**
   * Switcher link styles.
   *
   * @var array
   */
  protected $switcherLinkStyles = [];

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $account;

  /**
   * Indicator constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Site\Settings $settings
   *   The settings service.
   * @param \Drupal\Core\Session\AccountProxyInterface $account
   *   The current user.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    StateInterface $state,
    Settings $settings,
    AccountProxyInterface $account,
  ) {
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->state = $state;
    $this->settings = $settings;
    $this->account = $account;
    $this->activeEnvironment = $this->configFactory->get('environment_indicator.indicator');
    $this->activeConfig = $this->configFactory->get('environment_indicator.settings');
  }

  /**
   * Gets the active environment indicator configuration.
   *
   * @return \Drupal\Core\Config\ImmutableConfig
   *   The active environment indicator configuration.
   */
  public function getActiveEnvironment(): ImmutableConfig {
    return $this->activeEnvironment;
  }

  /**
   * Gets the active configuration for the environment indicator.
   *
   * @return \Drupal\Core\Config\ImmutableConfig
   *   The active configuration settings.
   */
  public function getActiveConfig(): ImmutableConfig {
    return $this->activeConfig;
  }

  /**
   * Gets the current release string based on version ID config and fallback.
   *
   * @return string|null
   *   The current release identifier, or NULL if not set.
   */
  public function getCurrentRelease(): ?string {
    if (!$this->account->hasPermission('view environment indicator current release')) {
      return NULL;
    }
    $config = $this->getActiveConfig();
    $primary = $config->get('version_identifier') ?? 'environment_indicator_current_release';
    $fallback = $config->get('version_identifier_fallback') ?? 'deployment_identifier';
    return $this->getVersionIdentifier($primary)
      ?? ($primary !== $fallback ? $this->getVersionIdentifier($fallback) : NULL);
  }

  /**
   * Resolves a version identifier by type.
   *
   * @param string $type
   *   The type of version identifier to retrieve.
   *
   * @return string|null
   *   The version identifier string, or NULL if not applicable.
   */
  protected function getVersionIdentifier(string $type): ?string {
    switch ($type) {
      case 'environment_indicator_current_release':
        return (string) $this->state->get('environment_indicator.current_release');

      case 'deployment_identifier':
        return (string) $this->settings->get('deployment_identifier');

      case 'drupal_version':
        return \Drupal::VERSION;

      default:
        return NULL;
    }
  }

  /**
   * Returns the combined label for the current environment.
   *
   * @return string|null
   *   The title string combining the environment name and current release,
   *   or NULL if no environment is set.
   */
  public function getTitle(): ?string {
    $environment = $this->activeEnvironment->get('name');
    $release = $this->getCurrentRelease();
    return $environment ? ($release ? "($release) $environment" : $environment) : NULL;
  }

  /**
   * Builds an array of links to switch environments.
   *
   * @return array[]
   *   A render array of link definitions for each active environment.
   */
  public function getLinks(): array {
    /** @var \Drupal\environment_indicator\Entity\EnvironmentIndicator[] $entities */
    $entities = $this->entityTypeManager->getStorage('environment_indicator')->loadMultiple();
    $current = Url::fromRoute('<current>');
    $current_path = $current->toString();
    $url = parse_url($current_path);
    $path = $url['path'];
    if (isset($url['query'])) {
      $path .= '?' . $url['query'];
    }
    $links = [];
    foreach ($entities as $entity) {
      if (!$entity->status() || empty($entity->getUrl())) {
        continue;
      }
      $links[$entity->id()] = [
        'attributes' => [
          'class' => array_merge(['environment-indicator'], [$entity->getClassName()]),
          'title' => $this->t('Opens the current page in the selected environment.'),
        ],
        'title' => $this->t('Open on @label', ['@label' => $entity->label()]),
        'url' => Url::fromUri($entity->getUrl() . $path),
        'type' => 'link',
        'weight' => $entity->getWeight(),
      ];
    }
    uasort($links, ['Drupal\Component\Utility\SortArray', 'sortByWeightElement']);

    return $links;
  }

  /**
   * Gets all link styles for switcher environments.
   *
   * @param string $selector
   *   The selector to use for styling.
   *   Defaults to '.environment-switcher-container'.
   *
   * @return array[]
   *   A list of html_head attachment arrays.
   */
  public function getAllSwitcherLinkStyles(string $selector = '.environment-switcher-container'): array {
    if ($this->switcherLinkStyles !== []) {
      return $this->switcherLinkStyles;
    }

    $styles = [];
    /** @var \Drupal\environment_indicator\Entity\EnvironmentIndicator[] $entities */
    $entities = $this->entityTypeManager->getStorage('environment_indicator')->loadMultiple();
    foreach ($entities as $entity) {
      $styles[] = [
        [
          '#tag' => 'style',
          '#value' => $entity->getLinkStyle($selector),
        ],
        'environment_indicator_style_' . $entity->id(),
      ];
    }

    $this->switcherLinkStyles = $styles;
    return $styles;
  }

  /**
   * Returns cache tags related to the switcher list.
   *
   * @return string[]
   *   An array of cache tags.
   */
  public function getCacheTags(): array {
    return $this->entityTypeManager
      ->getDefinition('environment_indicator')
      ->getListCacheTags();
  }

}
