<?php

declare(strict_types=1);

namespace Drupal\Tests\environment_indicator_toolbar\Functional;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests for Environment Indicator.
 *
 * @group environment_indicator
 */
class EnvironmentIndicatorToolbarTest extends BrowserTestBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['environment_indicator_toolbar', 'toolbar'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with permission to see the environment indicator.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $privilegedUser;

  /**
   * A user without permission to see the environment indicator.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $unprivilegedUser;

  /**
   * The path to the environment_indicator module.
   *
   * @var string
   */
  protected string $environmentIndicatorModulePath;

  /**
   * The path to the environment_indicator module.
   *
   * @var string
   */
  protected string $environmentIndicatorToolbarModulePath;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    // Retrieve the dynamic module path.
    $moduleHandler = \Drupal::service('extension.list.module');
    $this->environmentIndicatorModulePath = $moduleHandler->getPath('environment_indicator');
    $this->environmentIndicatorToolbarModulePath = $moduleHandler->getPath('environment_indicator_toolbar');

    // Disable CSS preprocessing.
    $config = $this->config('system.performance');
    $config->set('css.preprocess', FALSE)->save();

    // Create users.
    $this->privilegedUser = $this->drupalCreateUser(['access environment indicator', 'access toolbar']);
    $this->unprivilegedUser = $this->drupalCreateUser();
  }

  /**
   * Tests that the element appears in the page top region.
   *
   * This tests the toolbar module is enabled and the toolbar integration is
   * disabled.
   *
   * The environment indicator should never appear if the environment
   * indicator toolbar module is enabled.
   *
   * This also tests that the correct libraries are loaded.
   */
  public function testEnvironmentIndicatorVisibilityWithToolBarSettingDisabled(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Red Green Environment')
      ->set('fg_color', 'green')
      ->set('bg_color', 'red')
      ->save();
    $settings = $this->config('environment_indicator.settings');
    $settings->set('toolbar_integration', ['toolbar' => 0])
      ->save();
    // Clear drupal cache.
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet('<front>');
    $session = $this->assertSession();
    $session->pageTextContains('Red Green Environment');
    $session->elementNotExists('css', '#environment-indicator');
    $session->elementNotExists('css', "link[href*='{$this->environmentIndicatorModulePath}/css/environment_indicator.css']");
    $session->elementExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/toolbar.css']");
  }

  /**
   * Tests that the element does not appear in the page top region.
   *
   * If the toolbar module is enabled and the toolbar integration is enabled,
   * the environment indicator should not appear in the page top region.
   *
   * This also tests that the correct libraries are loaded.
   */
  public function testEnvironmentIndicatorVisibilityWithToolBarSettingEnabled(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Test Environment')
      ->set('fg_color', '#000000')
      ->set('bg_color', '#ffffff')
      ->save();
    $settings = $this->config('environment_indicator.settings');
    $settings->set('toolbar_integration', ['toolbar' => 'toolbar'])
      ->save();
    // Clear drupal cache.
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet('<front>');
    $this->assertSession()->elementNotExists('css', '#environment-indicator');
    $this->assertSession()->elementNotExists('css', "link[href*='{$this->environmentIndicatorModulePath}/css/environment_indicator.css']");
    $this->assertSession()->elementExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/toolbar.css']");
    $this->assertSession()->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/tinycon.min.js']");
    $this->assertSession()->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/favicon.js']");
  }

  /**
   * Tests that CSS selectors that environment indicator uses exist.
   */
  public function testEnvironmentIndicatorToolbarIntegration(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Test Environment')
      ->set('fg_color', '#000000')
      ->set('bg_color', '#ffffff')
      ->save();
    // Clear drupal cache.
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet('<front>');
    $session = $this->assertSession();
    $session->pageTextContains('Test Environment');
    $session->elementExists('css', 'nav[data-environment-indicator].toolbar-bar');
    $session->elementNotExists('css', '#environment-indicator');
    $session->elementNotExists('css', '#environment-indicator');
    $session->elementNotExists('css', "link[href*='{$this->environmentIndicatorModulePath}/css/environment_indicator.css']");
    $session->elementExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/toolbar.css']");
    $session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/tinycon.min.js']");
    $session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/favicon.js']");
    // Change configuration values.
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Development Environment')
      ->set('fg_color', '#efefef')
      ->set('bg_color', '#12285f')
      ->save();
    // Clear drupal cache.
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->drupalGet('<front>');
    // Assert that the indicator exists with the expected attributes.
    $session->elementExists('css', 'nav[data-environment-indicator].toolbar-bar');
    $session->elementNotExists('css', '#environment-indicator');
    $session->elementNotExists('css', '#environment-indicator');
    $session->elementNotExists('css', "link[href*='{$this->environmentIndicatorModulePath}/css/environment_indicator.css']");
    $session->elementExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/toolbar.css']");
    $session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/tinycon.min.js']");
    $session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/favicon.js']");
  }

}
