<?php

declare(strict_types=1);

namespace Drupal\Tests\environment_indicator\Functional;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests for Environment Indicator.
 *
 * @group environment_indicator
 */
class EnvironmentIndicatorTest extends BrowserTestBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'environment_indicator',
    'environment_indicator_ui',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with permission to see the environment indicator.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $privilegedUser;

  /**
   * A user without permission to see the environment indicator.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $unprivilegedUser;

  /**
   * A user with permission to see the current release.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $currentReleaseUser;

  /**
   * A user with permission to create environment switchers.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $environmentIndicatorAdministrator;

  /**
   * The path to the environment_indicator module.
   *
   * @var string
   */
  protected string $environmentIndicatorModulePath;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    // Retrieve the dynamic module path.
    $moduleHandler = \Drupal::service('extension.list.module');
    $this->environmentIndicatorModulePath = $moduleHandler->getPath('environment_indicator');
    $settings = $this->config('environment_indicator.settings');
    $settings->set('toolbar_integration', [])->save();
    $this->state = $this->container->get('state');
    $this->resetAll();
    // Create users.
    $this->privilegedUser = $this->drupalCreateUser(['access environment indicator']);
    $this->currentReleaseUser = $this->drupalCreateUser([
      'access environment indicator',
      'view environment indicator current release',
    ]);
    $this->unprivilegedUser = $this->drupalCreateUser();
  }

  /**
   * Tests that the environment indicator appears in the page top region.
   *
   * This test verifies that the environment indicator appears in the page top
   * region with the expected attributes.
   */
  public function testVisibilityWithPermissions(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Red Green Environment')
      ->set('fg_color', 'green')
      ->set('bg_color', 'red')
      ->save();
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->resetAll();
    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet('<front>');
    $assert_session = $this->assertSession();
    $assert_session->pageTextContains('Red Green Environment');
    $assert_session->elementExists('css', '#environment-indicator');
    $element = $this->getSession()->getPage()->find('css', '#environment-indicator');
    $this->assertNotNull($element, 'Environment indicator element found.');
    $this->assertNotEmpty($element->getAttribute('style'), 'Style attribute should not be empty.');
    $this->assertStringContainsString('background-color: red', $element->getAttribute('style'));
    $this->assertStringContainsString('color: green', $element->getAttribute('style'));
    $assert_session->elementExists("css", "link[href*='{$this->environmentIndicatorModulePath}/css/environment_indicator.css']");
    // Change configuration values.
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Test Environment')
      ->set('fg_color', '#ffffff')
      ->set('bg_color', '#000000')
      ->save();
    // Clear drupal cache.
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->resetAll();
    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet('<front>');
    $assert_session = $this->assertSession();
    $assert_session->pageTextContains('Test Environment');
    $assert_session->elementExists('css', '#environment-indicator');
    $element = $this->getSession()->getPage()->find('css', '#environment-indicator');
    $assert_session->elementExists('css', 'style[data-environment-indicator-styles]');
    $activeEnvironmentStyles = $this->getSession()->getPage()->find('css', 'style[data-environment-indicator-styles]');
    $this->assertSame($activeEnvironmentStyles->getText(), 'html body { --environment-indicator-color: #ffffff; --environment-indicator-background-color: #000000; }');
    $this->assertNotNull($element, 'Environment indicator element found.');
    $this->assertNotEmpty($element->getAttribute('style'), 'Style attribute should not be empty.');
    $this->assertStringContainsString('background-color: #000000', $element->getAttribute('style'));
    $this->assertStringContainsString('color: #ffffff', $element->getAttribute('style'));

    // Change configuration values.
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Development Environment')
      ->set('fg_color', '#efefef')
      ->set('bg_color', '#12285f')
      ->save();
    // Clear drupal cache.
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->resetAll();
    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet('<front>');
    // Assert that the environment indicator exists with
    // the expected attributes.
    $assert_session->elementExists('css', '#environment-indicator');
    $element = $this->getSession()->getPage()->find('css', '#environment-indicator');
    $this->assertNotNull($element, 'Environment indicator element found.');
    $this->assertNotEmpty($element->getAttribute('style'), 'Style attribute should not be empty.');
    $this->assertStringContainsString('background-color: #12285f', $element->getAttribute('style'));
    $this->assertStringContainsString('color: #efefef', $element->getAttribute('style'));
    $assert_session->pageTextContains('Development Environment');
    $assert_session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/tinycon.min.js']");
    $assert_session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/favicon.js']");
  }

  /**
   * Tests visibility for unauthorized users.
   */
  public function testVisibilityWithoutPermissions(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Test Environment')
      ->set('fg_color', '#000000')
      ->set('bg_color', '#ffffff')
      ->save();
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->resetAll();
    $this->drupalLogin($this->drupalCreateUser());
    $this->drupalGet('<front>');
    $assert_session = $this->assertSession();
    $assert_session->elementNotExists('css', '#environment-indicator');
    $assert_session->elementNotExists("css", "link[href*='{$this->environmentIndicatorModulePath}/css/environment_indicator.css']");
    $assert_session->elementNotExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/tinycon.min.js']");
    $assert_session->elementNotExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/favicon.js']");

  }

  /**
   * Tests the indicator with the default configuration.
   */
  public function testIndicatorDefaultConfiguration(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Indicator Environment')
      ->set('fg_color', '#000000')
      ->set('bg_color', '#ffffff')
      ->save();

    $this->state->set('environment_indicator.current_release', 'v1.2.44');
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);

    $this->resetAll();
    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet('<front>');
    $assert_session = $this->assertSession();
    $element = $this->getSession()->getPage()->find('css', '#environment-indicator');
    $this->assertNotNull($element, 'Environment indicator element found.');
    $this->assertNotEmpty($element->getAttribute('style'), 'Style attribute should not be empty.');
    $this->assertStringContainsString('background-color: #ffffff', $element->getAttribute('style'));
    $this->assertStringContainsString('color: #000000', $element->getAttribute('style'));
    $assert_session->elementNotExists('css', 'div#environment-indicator span.description');
    $assert_session->pageTextNotContains('v1.2.44');
    $assert_session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/tinycon.min.js']");
    $assert_session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/favicon.js']");
    $this->drupalLogin($this->currentReleaseUser);
    $this->drupalGet('<front>');
    $assert_session = $this->assertSession();
    $assert_session->elementExists('css', 'div#environment-indicator span.description');
    $assert_session->pageTextContains('v1.2.44');
  }

  /**
   * Tests that the environment indicator does not show when the name is empty.
   */
  public function testNoIndicatorWhenNameIsEmpty(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', '')
      ->set('fg_color', '#000000')
      ->set('bg_color', '#ffffff')
      ->save();
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet('<front>');
    $assert_session = $this->assertSession();
    $assert_session->elementNotExists('css', '#environment-indicator');
    $assert_session->elementNotExists("css", "link[href*='{$this->environmentIndicatorModulePath}/css/environment_indicator.css']");
    $assert_session->elementNotExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/tinycon.min.js']");
    $assert_session->elementNotExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/favicon.js']");
  }

  /**
   * Tests that an integer in state does not break the environment indicator.
   */
  public function testIntegerInState(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'State Integer Test')
      ->set('fg_color', '#123456')
      ->set('bg_color', '#abcdef')
      ->save();

    // Set an integer version identifier.
    $this->state->set('environment_indicator.current_release', 12345);

    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->resetAll();

    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet('<front>');
    $assert_session = $this->assertSession();
    $element = $this->getSession()->getPage()->find('css', '#environment-indicator');
    $this->assertNotNull($element, 'Environment indicator element not found.');
    $this->assertStringContainsString('background-color: #abcdef', $element->getAttribute('style'));
    $this->assertStringContainsString('color: #123456', $element->getAttribute('style'));
    $this->assertSession()->pageTextNotContains('12345');
    $this->drupalLogin($this->currentReleaseUser);
    $this->drupalGet('<front>');
    // The integer should be rendered as text in the indicator output.
    $assert_session->pageTextContains('12345');
    $assert_session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/tinycon.min.js']");
    $assert_session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/favicon.js']");
  }

  /**
   * Tests the current release visibility based on permissions.
   */
  public function testCurrentReleasePermission(): void {
    // Set environment name and current release.
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Permission Test Environment')
      ->set('fg_color', '#111111')
      ->set('bg_color', '#eeeeee')
      ->save();
    $this->state->set('environment_indicator.current_release', 'v9.9.9');
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->drupalLogin($this->currentReleaseUser);
    $this->drupalGet('<front>');
    $this->assertSession()->pageTextContains('v9.9.9');
    // User with only 'access environment indicator': should not see release.
    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet('<front>');
    $this->assertSession()->pageTextNotContains('v9.9.9');
  }

}
