<?php

declare(strict_types=1);

namespace Drupal\image_effects\Plugin\ImageEffect;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Image\ImageInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\image\Attribute\ImageEffect;
use Drupal\image\ConfigurableImageEffectBase;

/**
 * Blur image using the Gaussian function.
 */
#[ImageEffect(
  id: 'image_effects_gaussian_blur',
  label: new TranslatableMarkup('Gaussian blur'),
  description: new TranslatableMarkup('Blur the image with a Gaussian operator.'),
)]
class GaussianBlurImageEffect extends ConfigurableImageEffectBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'radius' => 3,
      'sigma' => NULL,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function getSummary() {
    return [
      '#theme' => 'image_effects_gaussian_blur_summary',
      '#data' => $this->configuration,
    ] + parent::getSummary();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['info'] = [
      '#type'  => 'details',
      '#title' => $this->t('Information'),
    ];
    $form['info']['help'] = [
      '#markup' => $this->t("A Gaussian blur is generated by replacing each pixel's color values with the average of the surrounding pixels' colors. This region is a circle whose radius is given by the 'radius' argument. Thus, a larger 'radius' will yield a blurrier image. This average is not a simple mean of the values. Instead, values are weighted using the Gaussian function (roughly a bell curve centered around the destination pixel) giving it much more influence on the result than its neighbours. Thus, a fatter curve will give the center pixel more weight and make the image less  blurry; lower 'sigma' values will yield flatter curves."),
    ];
    $form['radius'] = [
      '#type' => 'number',
      '#title' => $this->t('Radius'),
      '#description'  => $this->t('The radius of the blur, in pixels. <strong>The larger the radius, the slower the operation is.</strong> However, too small radius can lead to undesired aliasing effects.'),
      '#default_value' => $this->configuration['radius'],
      '#field_suffix' => $this->t('px'),
      '#required' => TRUE,
      '#size' => 3,
      '#min' => 1,
      '#max' => 15,
    ];
    $form['sigma'] = [
      '#type' => 'number',
      '#title' => $this->t('Sigma'),
      '#description'  => $this->t("Represents the 'fatness' of the Gaussian curve. Lower values mean less blurry image. Leave blank to use toolkit's default. As a guideline, 2/3 of the radius is a good starting point."),
      '#default_value' => $this->configuration['sigma'],
      '#required' => FALSE,
      '#size' => 5,
      '#min' => 0.1,
      '#max' => 15,
      '#step' => 0.1,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);
    $radius = (int) $form_state->getValue('radius');
    $sigma = (float) $form_state->getValue('sigma');
    if ($radius < $sigma) {
      $form_state->setError($form, $this->t("The value of <em>sigma</em> must be lower than the <em>radius</em>."));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    $this->configuration['radius'] = $form_state->getValue('radius');
    $this->configuration['sigma'] = $form_state->getValue('sigma');
  }

  /**
   * {@inheritdoc}
   */
  public function applyEffect(ImageInterface $image) {
    return $image->apply('gaussian_blur', [
      'radius' => $this->configuration['radius'],
      'sigma' => $this->configuration['sigma'],
    ]);
  }

}
