<?php

declare(strict_types=1);

namespace Drupal\Tests\image_effects\Functional\Effect;

use Drupal\imagemagick\ArgumentMode;
use Drupal\Tests\image_effects\Functional\ImageEffectsTestBase;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

/**
 * Convolution sharpen effect test.
 */
#[Group('image_effects')]
class ConvolutionSharpenTest extends ImageEffectsTestBase {

  /**
   * {@inheritdoc}
   */
  public static function providerToolkits(): array {
    $toolkits = parent::providerToolkits();
    // @todo This effect does not work on GraphicsMagick.
    unset($toolkits['ImageMagick-graphicsmagick']);
    return $toolkits;
  }

  /**
   * Convolution sharpen effect test.
   *
   * @param string $toolkit_id
   *   The id of the toolkit to set up.
   * @param string $toolkit_config
   *   The config object of the toolkit to set up.
   * @param array $toolkit_settings
   *   The settings of the toolkit to set up.
   */
  #[DataProvider('providerToolkits')]
  public function testConvolutionSharpenEffect(string $toolkit_id, string $toolkit_config, array $toolkit_settings): void {
    $this->changeToolkit($toolkit_id, $toolkit_config, $toolkit_settings);

    $original_uri = $this->getTestImageCopyUri('core/tests/fixtures/files/image-test.png');
    $derivative_uri = 'public://test-images/image-test-derived.png';

    // Add convolution sharpen effect to the test image style.
    $effect = [
      'id' => 'image_effects_convolution_sharpen',
      'data' => [
        'level' => 10,
      ],
    ];
    $uuid = $this->addEffectToTestStyle($effect);

    // Apply the operation, via the effect.
    $image = $this->imageFactory->get($original_uri);
    $effect = $this->testImageStyle->getEffect($uuid);
    $effect->applyEffect($image);

    // Toolkit-specific tests.
    switch ($this->imageFactory->getToolkitId()) {
      case 'gd':
        // For the GD toolkit, just test derivative image is valid.
        $image->save($derivative_uri);
        $derivative_image = $this->imageFactory->get($derivative_uri);
        $this->assertTrue($derivative_image->isValid());
        break;

      case 'imagemagick':
        // For the Imagemagick toolkit, check the command line argument has
        // been formatted properly.
        /** @var \Drupal\imagemagick\Plugin\ImageToolkit\ImagemagickToolkit $toolkit */
        $toolkit = $image->getToolkit();
        $this->assertStringContainsString("[-morphology] [Convolve] [3x3:-0.1,-0.1,-0.1 -0.1,1.8,-0.1 -0.1,-0.1,-0.1]", $toolkit->arguments()->toDebugString(ArgumentMode::PostSource));
        break;

    }

    // Remove effect.
    $this->removeEffectFromTestStyle($uuid);

  }

}
