<?php

declare(strict_types=1);

namespace Drupal\trash\Handler;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Entity\EntityInterface;
use Drupal\trash\Exception\UnrestorableEntityException;
use Drupal\trash\PathAlias\PathAliasIntegrationTrait;
use Drupal\trash\TrashManagerInterface;

/**
 * Provides the default trash handler.
 */
class DefaultTrashHandler implements TrashHandlerInterface {

  use StringTranslationTrait;
  use PathAliasIntegrationTrait;

  /**
   * The ID of the entity type managed by this handler.
   */
  protected string $entityTypeId;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The trash manager.
   */
  protected TrashManagerInterface $trashManager;

  /**
   * Tracks whether validation has already been performed for an entity.
   *
   * This prevents duplicate validation when validateRestore() is called
   * explicitly (e.g., from form validation) before preTrashRestore().
   *
   * @var array<string, bool>
   */
  protected array $validatedEntities = [];

  /**
   * {@inheritdoc}
   */
  public function preTrashDelete(EntityInterface $entity): void {}

  /**
   * {@inheritdoc}
   */
  public function postTrashDelete(EntityInterface $entity): void {
    // Automatically delete associated path aliases to match core's behavior.
    $this->deleteAssociatedPathAliases($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function validateRestore(EntityInterface $entity): void {
    $entity_key = $entity->getEntityTypeId() . ':' . $entity->id();
    $this->validatedEntities[$entity_key] = TRUE;

    // Run entity validation for fieldable entities to check for conflicts.
    if ($entity instanceof FieldableEntityInterface) {
      $violations = $entity->validate();
      if ($violations->count() > 0) {
        // Get the first violation message for the exception.
        $message = $violations->get(0)->getMessage();
        throw new UnrestorableEntityException((string) $message);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function preTrashRestore(EntityInterface $entity): void {}

  /**
   * {@inheritdoc}
   */
  public function postTrashRestore(EntityInterface $entity, int|string $deleted_timestamp): void {
    // Automatically restore associated path aliases.
    $this->restoreAssociatedPathAliases($entity, $deleted_timestamp);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteFormAlter(array &$form, FormStateInterface $form_state, bool $multiple = FALSE): void {}

  /**
   * {@inheritdoc}
   */
  public function restoreFormAlter(array &$form, FormStateInterface $form_state): void {}

  /**
   * {@inheritdoc}
   */
  public function purgeFormAlter(array &$form, FormStateInterface $form_state): void {}

  /**
   * {@inheritdoc}
   */
  public function setEntityTypeId(string $entity_type_id): static {
    $this->entityTypeId = $entity_type_id;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setEntityTypeManager(EntityTypeManagerInterface $entity_type_manager): static {
    $this->entityTypeManager = $entity_type_manager;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setTrashManager(TrashManagerInterface $trash_manager): static {
    $this->trashManager = $trash_manager;
    return $this;
  }

}
